<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Artisan;

class InstallMintCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'install:mint {--fresh : Run fresh migration} {--force : Force the operation to run when in production}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Install and setup Mint Mobile Shop System with admin user and sample data';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('');
        $this->info('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
        $this->info('🚀 MINT MOBILE SHOP - INSTALLATION');
        $this->info('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
        $this->info('');

        // Confirm if running in production
        if ($this->laravel->environment('production') && !$this->option('force')) {
            $this->error('⚠️  Application is in production mode!');
            if (!$this->confirm('Do you really want to continue?', false)) {
                $this->warn('Installation cancelled.');
                return 0;
            }
        }

        // Check if fresh migration is requested
        $fresh = $this->option('fresh');

        if ($fresh) {
            $this->warn('⚠️  Running fresh migration will delete all existing data!');
            if (!$this->confirm('Are you sure you want to continue?', true)) {
                $this->warn('Installation cancelled.');
                return 0;
            }
        }

        try {
            // Step 1: Clear caches
            $this->info('🔄 Clearing caches...');
            Artisan::call('config:clear');
            Artisan::call('cache:clear');
            Artisan::call('route:clear');
            Artisan::call('view:clear');
            $this->info('✅ Caches cleared successfully!');
            $this->newLine();

            // Step 2: Run migrations
            $this->info('🗄️  Running database migrations...');
            if ($fresh) {
                Artisan::call('migrate:fresh', ['--force' => true]);
                $this->info('✅ Fresh migrations completed!');
            } else {
                Artisan::call('migrate', ['--force' => true]);
                $this->info('✅ Migrations completed!');
            }
            $this->newLine();

            // Step 3: Run seeders
            $this->info('🌱 Seeding database with sample data...');
            Artisan::call('db:seed', [
                '--class' => 'AdminSeeder',
                '--force' => true
            ]);
            Artisan::call('db:seed', [
                '--class' => 'InventorySeeder',
                '--force' => true
            ]);
            $this->newLine();

            // Step 4: Optimize application
            $this->info('⚡ Optimizing application...');
            Artisan::call('config:cache');
            Artisan::call('route:cache');
            $this->info('✅ Application optimized!');
            $this->newLine();

            // Step 5: Show summary
            $this->displaySummary();

            return 0;

        } catch (\Exception $e) {
            $this->error('❌ Installation failed!');
            $this->error($e->getMessage());
            return 1;
        }
    }

    /**
     * Display installation summary
     */
    protected function displaySummary()
    {
        $this->info('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
        $this->info('🎉 INSTALLATION COMPLETED SUCCESSFULLY!');
        $this->info('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
        $this->newLine();

        $this->info('📊 System Status:');
        $this->table(
            ['Component', 'Status'],
            [
                ['Database', '✅ Connected'],
                ['Migrations', '✅ Completed'],
                ['Sample Data', '✅ Seeded'],
                ['Admin User', '✅ Created'],
                ['Application', '✅ Ready'],
            ]
        );

        $this->newLine();
        $this->info('🔐 Admin Login Credentials:');
        $this->line('   Email:    <fg=cyan>admin@admin.com</>');
        $this->line('   Password: <fg=cyan>admin</>');
        $this->newLine();

        $this->info('📝 Sample Data Created:');
        $this->line('   • 8 Mobile Brands');
        $this->line('   • 5 Product Categories');
        $this->line('   • 2 Suppliers');
        $this->line('   • 3 Customers');
        $this->line('   • System Settings Configured');
        $this->newLine();

        $this->info('🚀 Next Steps:');
        $this->line('   1. Test login: POST /api/login');
        $this->line('   2. Import Postman collection');
        $this->line('   3. Start adding inventory');
        $this->line('   4. Create your first sale!');
        $this->newLine();

        $this->info('📚 API Documentation:');
        $this->line('   Base URL: ' . config('app.url') . '/api');
        $this->line('   Total Endpoints: 100+');
        $this->newLine();

        $this->info('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
        $this->info('✨ Happy Selling!');
        $this->info('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
        $this->newLine();
    }
}