<?php

namespace App\Exports;

use Maatwebsite\Excel\Concerns\FromArray;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithTitle;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Alignment;

class ReportExport implements FromArray, WithHeadings, WithStyles, WithTitle
{
    protected $reportType;
    protected $data;

    public function __construct(string $reportType, array $data)
    {
        $this->reportType = $reportType;
        $this->data = $data;
    }

    /**
     * Return array of data
     */
    public function array(): array
    {
        return $this->formatDataForExcel();
    }

    /**
     * Return headings
     */
    public function headings(): array
    {
        return $this->getHeadings();
    }

    /**
     * Return sheet title
     */
    public function title(): string
    {
        return substr($this->getReportTitle(), 0, 31); // Excel limit
    }

    /**
     * Apply styles
     */
    public function styles(Worksheet $sheet)
    {
        return [
            1 => [
                'font' => ['bold' => true, 'size' => 12],
                'fill' => [
                    'fillType' => Fill::FILL_SOLID,
                    'startColor' => ['rgb' => '4472C4']
                ],
                'font' => ['color' => ['rgb' => 'FFFFFF'], 'bold' => true],
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER
                ]
            ],
        ];
    }

    /**
     * Format data for Excel export
     */
    private function formatDataForExcel(): array
    {
        return match ($this->reportType) {
            'sales-summary' => $this->formatSalesSummary(),
            'sales-by-customer' => $this->formatSalesByCustomer(),
            'sales-by-product' => $this->formatSalesByProduct(),
            'payment-received' => $this->formatPaymentReceived(),
            'purchase-summary' => $this->formatPurchaseSummary(),
            'purchase-by-supplier' => $this->formatPurchaseBySupplier(),
            'payment-made' => $this->formatPaymentMade(),
            'stock-summary' => $this->formatStockSummary(),
            'low-stock-alert' => $this->formatLowStockAlert(),
            'stock-movement' => $this->formatStockMovement(),
            'profit-loss' => $this->formatProfitLoss(),
            'cash-flow' => $this->formatCashFlow(),
            'tax-summary' => $this->formatTaxSummary(),
            'outstanding-receivables' => $this->formatOutstandingReceivables(),
            'outstanding-payables' => $this->formatOutstandingPayables(),
            default => []
        };
    }

    /**
     * Get headings based on report type
     */
    private function getHeadings(): array
    {
        return match ($this->reportType) {
            'sales-summary' => ['Invoice', 'Date', 'Customer', 'Total Amount', 'Paid', 'Due', 'Status'],
            'sales-by-customer' => ['Customer Name', 'Phone', 'Total Sales', 'Total Amount', 'Total Paid', 'Total Due'],
            'sales-by-product' => ['Product Name', 'Quantity Sold', 'Total Sales', 'Average Price'],
            'payment-received' => ['Date', 'Invoice', 'Customer', 'Amount', 'Payment Method'],
            'purchase-summary' => ['Invoice', 'Date', 'Supplier', 'Total Amount', 'Paid', 'Due', 'Status'],
            'purchase-by-supplier' => ['Supplier Name', 'Phone', 'Total Purchases', 'Total Amount', 'Total Paid', 'Total Due'],
            'payment-made' => ['Date', 'Invoice', 'Supplier', 'Amount', 'Payment Method'],
            'stock-summary' => ['Product Name', 'Brand', 'Model', 'Stock Quantity', 'Total Value', 'Avg Cost'],
            'low-stock-alert' => ['Product Name', 'Brand', 'Model', 'Current Stock', 'Status'],
            'stock-movement' => ['Date', 'Product', 'Type', 'Quantity', 'Value'],
            'profit-loss' => ['Metric', 'Amount'],
            'cash-flow' => ['Date', 'Cash In', 'Cash Out', 'Net'],
            'tax-summary' => ['Type', 'Amount'],
            'outstanding-receivables' => ['Invoice', 'Customer', 'Phone', 'Sale Date', 'Due Date', 'Amount', 'Paid', 'Due', 'Days Overdue', 'Status'],
            'outstanding-payables' => ['Invoice', 'Supplier', 'Phone', 'Purchase Date', 'Due Date', 'Amount', 'Paid', 'Due', 'Days Overdue', 'Status'],
            default => ['Data']
        };
    }

    /**
     * Get report title
     */
    private function getReportTitle(): string
    {
        return match ($this->reportType) {
            'sales-summary' => 'Sales Summary',
            'sales-by-customer' => 'Sales by Customer',
            'sales-by-product' => 'Sales by Product',
            'payment-received' => 'Payment Received',
            'purchase-summary' => 'Purchase Summary',
            'purchase-by-supplier' => 'Purchase by Supplier',
            'payment-made' => 'Payment Made',
            'stock-summary' => 'Stock Summary',
            'low-stock-alert' => 'Low Stock Alert',
            'stock-movement' => 'Stock Movement',
            'profit-loss' => 'Profit & Loss',
            'cash-flow' => 'Cash Flow',
            'tax-summary' => 'Tax Summary',
            'outstanding-receivables' => 'Outstanding Receivables',
            'outstanding-payables' => 'Outstanding Payables',
            default => 'Report'
        };
    }

    // Format methods for each report type

    private function formatSalesSummary(): array
    {
        $rows = [];
        foreach ($this->data['sales'] ?? [] as $sale) {
            $rows[] = [
                $sale['invoice_number'] ?? '',
                $sale['sale_date'] ?? '',
                $sale['customer']['name'] ?? 'Walk-in',
                $sale['total_amount'] ?? 0,
                $sale['paid_amount'] ?? 0,
                $sale['due_amount'] ?? 0,
                $sale['status'] ?? ''
            ];
        }
        return $rows;
    }

    private function formatSalesByCustomer(): array
    {
        $rows = [];
        foreach ($this->data['customers'] ?? [] as $customer) {
            $rows[] = [
                $customer['customer_name'] ?? '',
                $customer['customer_phone'] ?? '',
                $customer['total_sales'] ?? 0,
                $customer['total_amount'] ?? 0,
                $customer['total_paid'] ?? 0,
                $customer['total_due'] ?? 0
            ];
        }
        return $rows;
    }

    private function formatSalesByProduct(): array
    {
        $rows = [];
        foreach ($this->data['products'] ?? [] as $product) {
            $rows[] = [
                $product['product_name'] ?? '',
                $product['total_quantity'] ?? 0,
                $product['total_sales'] ?? 0,
                $product['average_price'] ?? 0
            ];
        }
        return $rows;
    }

    private function formatPaymentReceived(): array
    {
        $rows = [];
        foreach ($this->data['payments'] ?? [] as $payment) {
            $rows[] = [
                $payment['payment_date'] ?? '',
                $payment['invoice_number'] ?? '',
                $payment['customer_name'] ?? 'Walk-in',
                $payment['amount'] ?? 0,
                $payment['payment_method'] ?? ''
            ];
        }
        return $rows;
    }

    private function formatPurchaseSummary(): array
    {
        $rows = [];
        foreach ($this->data['purchases'] ?? [] as $purchase) {
            $rows[] = [
                $purchase['invoice_number'] ?? '',
                $purchase['purchase_date'] ?? '',
                $purchase['supplier']['name'] ?? 'Unknown',
                $purchase['total_amount'] ?? 0,
                $purchase['paid_amount'] ?? 0,
                $purchase['due_amount'] ?? 0,
                $purchase['status'] ?? ''
            ];
        }
        return $rows;
    }

    private function formatPurchaseBySupplier(): array
    {
        $rows = [];
        foreach ($this->data['suppliers'] ?? [] as $supplier) {
            $rows[] = [
                $supplier['supplier_name'] ?? '',
                $supplier['supplier_phone'] ?? '',
                $supplier['total_purchases'] ?? 0,
                $supplier['total_amount'] ?? 0,
                $supplier['total_paid'] ?? 0,
                $supplier['total_due'] ?? 0
            ];
        }
        return $rows;
    }

    private function formatPaymentMade(): array
    {
        $rows = [];
        foreach ($this->data['payments'] ?? [] as $payment) {
            $rows[] = [
                $payment['payment_date'] ?? '',
                $payment['invoice_number'] ?? '',
                $payment['supplier_name'] ?? 'Unknown',
                $payment['amount'] ?? 0,
                $payment['payment_method'] ?? ''
            ];
        }
        return $rows;
    }

    private function formatStockSummary(): array
    {
        $rows = [];
        foreach ($this->data['stock'] ?? [] as $item) {
            $rows[] = [
                $item['product_name'] ?? '',
                $item['brand'] ?? '',
                $item['model'] ?? '',
                $item['total_stock'] ?? 0,
                $item['total_value'] ?? 0,
                $item['average_cost'] ?? 0
            ];
        }
        return $rows;
    }

    private function formatLowStockAlert(): array
    {
        $rows = [];
        foreach ($this->data['low_stock_products'] ?? [] as $product) {
            $rows[] = [
                $product['product_name'] ?? '',
                $product['brand'] ?? '',
                $product['model'] ?? '',
                $product['current_stock'] ?? 0,
                $product['status'] ?? ''
            ];
        }
        return $rows;
    }

    private function formatStockMovement(): array
    {
        $rows = [];
        foreach ($this->data['movements'] ?? [] as $movement) {
            $rows[] = [
                $movement['date'] ?? '',
                $movement['product_name'] ?? '',
                $movement['type'] ?? '',
                $movement['quantity'] ?? 0,
                $movement['total_cost'] ?? 0
            ];
        }
        return $rows;
    }

    private function formatProfitLoss(): array
    {
        return [
            ['Revenue', $this->data['revenue'] ?? 0],
            ['Cost of Goods Sold', $this->data['cogs'] ?? 0],
            ['Gross Profit', $this->data['gross_profit'] ?? 0],
            ['Gross Profit Margin (%)', $this->data['gross_profit_margin'] ?? 0],
            ['Operating Expenses', $this->data['operating_expenses'] ?? 0],
            ['Net Profit', $this->data['net_profit'] ?? 0],
            ['Net Profit Margin (%)', $this->data['net_profit_margin'] ?? 0],
        ];
    }

    private function formatCashFlow(): array
    {
        return [
            ['Cash In', $this->data['cash_in'] ?? 0],
            ['Cash Out', $this->data['cash_out'] ?? 0],
            ['Net Cash Flow', $this->data['net_cash_flow'] ?? 0],
        ];
    }

    private function formatTaxSummary(): array
    {
        return [
            ['Sales Tax Collected', $this->data['sales_tax_collected'] ?? 0],
            ['Purchase Tax Paid', $this->data['purchase_tax_paid'] ?? 0],
            ['Net Tax Payable', $this->data['net_tax_payable'] ?? 0],
        ];
    }

    private function formatOutstandingReceivables(): array
    {
        $rows = [];
        foreach ($this->data['receivables'] ?? [] as $receivable) {
            $rows[] = [
                $receivable['invoice_number'] ?? '',
                $receivable['customer_name'] ?? '',
                $receivable['customer_phone'] ?? '',
                $receivable['sale_date'] ?? '',
                $receivable['due_date'] ?? '',
                $receivable['total_amount'] ?? 0,
                $receivable['paid_amount'] ?? 0,
                $receivable['due_amount'] ?? 0,
                $receivable['days_overdue'] ?? 0,
                $receivable['status'] ?? ''
            ];
        }
        return $rows;
    }

    private function formatOutstandingPayables(): array
    {
        $rows = [];
        foreach ($this->data['payables'] ?? [] as $payable) {
            $rows[] = [
                $payable['invoice_number'] ?? '',
                $payable['supplier_name'] ?? '',
                $payable['supplier_phone'] ?? '',
                $payable['purchase_date'] ?? '',
                $payable['due_date'] ?? '',
                $payable['total_amount'] ?? 0,
                $payable['paid_amount'] ?? 0,
                $payable['due_amount'] ?? 0,
                $payable['days_overdue'] ?? 0,
                $payable['status'] ?? ''
            ];
        }
        return $rows;
    }
}