<?php

namespace App\Http\Controllers\Api;

use App\Models\Accessory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Http\Controllers\Controller;

class AccessoryController extends Controller
{
    public function index(Request $request)
    {
        $query = Accessory::query()->with(['purchases' => function ($q) {
            $q->latest()->take(5);
        }]);

        // Filter by active status
        if ($request->has('is_active')) {
            $query->where('is_active', $request->is_active);
        }

        // Filter by low stock
        if ($request->has('low_stock') && $request->low_stock) {
            $query->lowStock();
        }

        // Search
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('sku', 'like', "%{$search}%")
                    ->orWhere('brand', 'like', "%{$search}%")
                    ->orWhere('category', 'like', "%{$search}%");
            });
        }

        // Filter by category
        if ($request->has('category')) {
            $query->where('category', $request->category);
        }

        $accessories = $query->orderBy('name')->get();

        return response()->json($accessories);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'category' => 'nullable|string|max:255',
            'brand' => 'nullable|string|max:255',
            'model' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'sku' => 'nullable|string|unique:accessories,sku',
            'purchase_price' => 'required|numeric|min:0',
            'selling_price' => 'required|numeric|min:0',
            'current_stock' => 'nullable|integer|min:0',
            'minimum_stock' => 'nullable|integer|min:0',
            'unit' => 'nullable|string|max:50',
            'is_active' => 'boolean'
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $accessory = Accessory::create($request->all());

        return response()->json([
            'message' => 'Accessory created successfully',
            'accessory' => $accessory
        ], 201);
    }

    public function show($id)
    {
        $accessory = Accessory::with(['purchases.supplier', 'stockMovements'])
            ->findOrFail($id);

        return response()->json($accessory);
    }

    public function update(Request $request, $id)
    {
        $accessory = Accessory::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'category' => 'nullable|string|max:255',
            'brand' => 'nullable|string|max:255',
            'model' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'sku' => 'nullable|string|unique:accessories,sku,' . $id,
            'purchase_price' => 'required|numeric|min:0',
            'selling_price' => 'required|numeric|min:0',
            'minimum_stock' => 'nullable|integer|min:0',
            'unit' => 'nullable|string|max:50',
            'is_active' => 'boolean'
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $accessory->update($request->all());

        return response()->json([
            'message' => 'Accessory updated successfully',
            'accessory' => $accessory
        ]);
    }

    public function destroy($id)
    {
        $accessory = Accessory::findOrFail($id);
        $accessory->delete();

        return response()->json([
            'message' => 'Accessory deleted successfully'
        ]);
    }

    public function lowStock()
    {
        $accessories = Accessory::lowStock()
            ->active()
            ->orderBy('current_stock')
            ->get();

        return response()->json($accessories);
    }

    public function categories()
    {
        $categories = Accessory::distinct()
            ->pluck('category')
            ->filter()
            ->values();

        return response()->json($categories);
    }

    public function availableForSale()
    {
        \Log::info('asss');
        $accessories = Accessory::where('is_active', 1)
            ->where('current_stock', '>', 0)
            ->select('id', 'name', 'brand', 'model', 'category', 'selling_price', 'current_stock', 'unit')
            ->orderBy('name')
            ->get();

        return response()->json($accessories);
    }
}
