<?php

namespace App\Http\Controllers\Api;


use App\Models\Accessory;
use App\Models\AccessoryStockMovement;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use App\Http\Controllers\Controller;

class AccessoryStockAdjustmentController extends Controller
{
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'accessory_id' => 'required|exists:accessories,id',
            'type' => 'required|in:adjustment,return',
            'quantity' => 'required|integer|not_in:0',
            'notes' => 'required|string'
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        try {
            DB::beginTransaction();

            $accessory = Accessory::findOrFail($request->accessory_id);
            $stockBefore = $accessory->current_stock;
            $stockAfter = $stockBefore + $request->quantity;

            // Validate stock won't go negative
            if ($stockAfter < 0) {
                return response()->json([
                    'message' => 'Insufficient stock for this adjustment'
                ], 400);
            }

            // Update accessory stock
            $accessory->update([
                'current_stock' => $stockAfter
            ]);

            // Create stock movement
            AccessoryStockMovement::create([
                'accessory_id' => $accessory->id,
                'type' => $request->type,
                'quantity' => $request->quantity,
                'stock_before' => $stockBefore,
                'stock_after' => $stockAfter,
                'notes' => $request->notes
            ]);

            DB::commit();

            return response()->json([
                'message' => 'Stock adjusted successfully',
                'accessory' => $accessory
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'Failed to adjust stock',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}