<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Account;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class AccountController extends Controller
{
    /**
     * Get all accounts
     */
    public function index(Request $request): JsonResponse
    {
        $query = Account::query();

        // Filter by type
        if ($request->has('type')) {
            $query->where('type', $request->type);
        }

        // Filter by active status
        if ($request->has('is_active')) {
            $query->where('is_active', $request->boolean('is_active'));
        }

        $accounts = $query->orderBy('is_default', 'desc')
            ->orderBy('name')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $accounts,
        ]);
    }

    /**
     * Create a new account
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'account_number' => 'nullable|string|max:255',
            'type' => 'required|in:cash,bank,mobile_money,credit_card,other',
            'bank_name' => 'nullable|string|max:255',
            'branch' => 'nullable|string|max:255',
            'currency' => 'nullable|string|size:3',
            'opening_balance' => 'nullable|numeric',
            'description' => 'nullable|string',
            'is_active' => 'boolean',
            'is_default' => 'boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        DB::beginTransaction();
        try {
            // If this is set as default, unset other defaults
            if ($request->is_default) {
                Account::where('is_default', true)->update(['is_default' => false]);
            }

            $account = Account::create([
                'name' => $request->name,
                'account_number' => $request->account_number,
                'type' => $request->type,
                'bank_name' => $request->bank_name,
                'branch' => $request->branch,
                'currency' => $request->currency ?? 'PKR',
                'opening_balance' => $request->opening_balance ?? 0,
                'current_balance' => $request->opening_balance ?? 0,
                'description' => $request->description,
                'is_active' => $request->is_active ?? true,
                'is_default' => $request->is_default ?? false,
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Account created successfully',
                'data' => $account,
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to create account: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get a single account with details
     */
    public function show($id): JsonResponse
    {
        $account = Account::find($id);

        if (!$account) {
            return response()->json([
                'success' => false,
                'message' => 'Account not found',
            ], 404);
        }

        // Get transaction stats
        $totalCredits = $account->transactions()
            ->where('direction', 'credit')
            ->where('status', 'completed')
            ->sum('amount');

        $totalDebits = $account->transactions()
            ->where('direction', 'debit')
            ->where('status', 'completed')
            ->sum('amount');

        $transactionCount = $account->transactions()->count();

        return response()->json([
            'success' => true,
            'data' => [
                'account' => $account,
                'stats' => [
                    'opening_balance' => $account->opening_balance,
                    'current_balance' => $account->current_balance,
                    'total_credits' => $totalCredits,
                    'total_debits' => $totalDebits,
                    'transaction_count' => $transactionCount,
                ],
            ],
        ]);
    }

    /**
     * Update an account
     */
    public function update(Request $request, $id): JsonResponse
    {
        $account = Account::find($id);

        if (!$account) {
            return response()->json([
                'success' => false,
                'message' => 'Account not found',
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'account_number' => 'nullable|string|max:255',
            'type' => 'required|in:cash,bank,mobile_money,credit_card,other',
            'bank_name' => 'nullable|string|max:255',
            'branch' => 'nullable|string|max:255',
            'currency' => 'nullable|string|size:3',
            'opening_balance' => 'nullable|numeric',
            'description' => 'nullable|string',
            'is_active' => 'boolean',
            'is_default' => 'boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        DB::beginTransaction();
        try {
            // If setting as default, unset other defaults
            if ($request->is_default) {
                Account::where('id', '!=', $id)
                    ->where('is_default', true)
                    ->update(['is_default' => false]);
            }

            $account->update([
                'name' => $request->name,
                'account_number' => $request->account_number,
                'type' => $request->type,
                'bank_name' => $request->bank_name,
                'branch' => $request->branch,
                'currency' => $request->currency ?? $account->currency,
                'opening_balance' => $request->opening_balance ?? $account->opening_balance,
                'description' => $request->description,
                'is_active' => $request->is_active ?? $account->is_active,
                'is_default' => $request->is_default ?? $account->is_default,
            ]);

            // Recalculate balance if opening balance changed
            if ($request->has('opening_balance')) {
                $account->updateBalance();
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Account updated successfully',
                'data' => $account,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to update account: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Delete an account (soft delete)
     */
    public function destroy($id): JsonResponse
    {
        $account = Account::find($id);

        if (!$account) {
            return response()->json([
                'success' => false,
                'message' => 'Account not found',
            ], 404);
        }

        // Check if account has transactions
        $hasTransactions = $account->transactions()->exists();
        if ($hasTransactions) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete account that has transactions. Please deactivate it instead.',
            ], 400);
        }

        // Don't allow deleting default account
        if ($account->is_default) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete default account. Please set another account as default first.',
            ], 400);
        }

        try {
            $account->delete();

            return response()->json([
                'success' => true,
                'message' => 'Account deleted successfully',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete account: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get transactions for an account
     */
    public function transactions(Request $request, $id): JsonResponse
    {
        $account = Account::find($id);

        if (!$account) {
            return response()->json([
                'success' => false,
                'message' => 'Account not found',
            ], 404);
        }

        $query = $account->transactions()->with(['category', 'creator']);

        // Filter by date range
        if ($request->has('start_date') && $request->has('end_date')) {
            $query->whereBetween('transaction_date', [$request->start_date, $request->end_date]);
        }

        // Filter by type
        if ($request->has('type')) {
            $query->where('type', $request->type);
        }

        // Filter by direction
        if ($request->has('direction')) {
            $query->where('direction', $request->direction);
        }

        $transactions = $query->orderBy('transaction_date', 'desc')
            ->orderBy('created_at', 'desc')
            ->paginate(50);

        return response()->json([
            'success' => true,
            'data' => $transactions,
        ]);
    }

    /**
     * Get account balance details
     */
    public function balance($id): JsonResponse
    {
        $account = Account::find($id);

        if (!$account) {
            return response()->json([
                'success' => false,
                'message' => 'Account not found',
            ], 404);
        }

        $calculatedBalance = $account->calculateBalance();
        
        // Get credits and debits
        $credits = $account->transactions()
            ->where('direction', 'credit')
            ->where('status', 'completed')
            ->sum('amount');

        $debits = $account->transactions()
            ->where('direction', 'debit')
            ->where('status', 'completed')
            ->sum('amount');

        return response()->json([
            'success' => true,
            'data' => [
                'account' => $account,
                'opening_balance' => $account->opening_balance,
                'current_balance' => $account->current_balance,
                'calculated_balance' => $calculatedBalance,
                'total_credits' => $credits,
                'total_debits' => $debits,
                'net_movement' => $credits - $debits,
            ],
        ]);
    }
}
