<?php

// app/Http/Controllers/Api/AttendanceController.php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Employee;
use App\Models\EmployeeAttendance;
use Carbon\Carbon;
use Illuminate\Http\Request;

class AttendanceController extends Controller
{
    /**
     * Clock in an employee
     */
    public function clockIn(Request $request)
    {
        $data = $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'notes'       => 'nullable|string',
            'source'      => 'nullable|string',
        ]);

        $employee = Employee::findOrFail($data['employee_id']);
        $today = Carbon::today();

        // Check if already clocked in today
        $existing = EmployeeAttendance::where('employee_id', $employee->id)
            ->where('work_date', $today->toDateString())
            ->first();

        if ($existing && $existing->clock_in_at) {
            return response()->json([
                'message' => 'Already clocked in today',
                'attendance' => $existing
            ], 422);
        }

        $attendance = EmployeeAttendance::firstOrCreate(
            [
                'employee_id' => $employee->id,
                'work_date'   => $today->toDateString(),
            ],
            [
                'clock_in_at' => now(),
                'status'      => 'open',
                'source'      => $data['source'] ?? 'web',
                'notes'       => $data['notes'] ?? null,
            ]
        );

        // If already exists but no clock in
        if (!$attendance->wasRecentlyCreated && !$attendance->clock_in_at) {
            $attendance->clock_in_at = now();
            $attendance->status = 'open';
            $attendance->save();
        }

        return response()->json([
            'message'    => 'Clock-in recorded successfully',
            'attendance' => $attendance,
        ]);
    }

    /**
     * Clock out an employee
     */
    public function clockOut(Request $request)
    {
        $data = $request->validate([
            'employee_id'   => 'required|exists:employees,id',
            'break_minutes' => 'nullable|integer|min:0',
        ]);

        $today = Carbon::today();

        $attendance = EmployeeAttendance::where('employee_id', $data['employee_id'])
            ->where('work_date', $today->toDateString())
            ->first();

        if (!$attendance || !$attendance->clock_in_at) {
            return response()->json(['message' => 'No open attendance for today'], 422);
        }

        if ($attendance->clock_out_at) {
            return response()->json(['message' => 'Already clocked out today'], 422);
        }

        $attendance->clock_out_at = now();
        $attendance->break_minutes = $data['break_minutes'] ?? ($attendance->break_minutes ?? 0);

        $minutes = $attendance->clock_in_at->diffInMinutes($attendance->clock_out_at);
        $attendance->total_minutes = max(0, $minutes - $attendance->break_minutes);
        $attendance->status = 'completed';
        $attendance->save();

        return response()->json([
            'message'    => 'Clock-out recorded successfully',
            'attendance' => $attendance,
        ]);
    }

    /**
     * Get attendance history
     */
    public function history(Request $request)
    {
        $data = $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'from'        => 'nullable|date',
            'to'          => 'nullable|date',
        ]);

        $query = EmployeeAttendance::where('employee_id', $data['employee_id'])
            ->orderByDesc('work_date');

        if (!empty($data['from'])) {
            $query->whereDate('work_date', '>=', $data['from']);
        }

        if (!empty($data['to'])) {
            $query->whereDate('work_date', '<=', $data['to']);
        }

        return $query->paginate(30);
    }

    /**
     * Show single attendance record
     */
    public function show($id)
    {
        $attendance = EmployeeAttendance::with('employee')->findOrFail($id);
        
        // Ensure datetime fields are properly formatted
        if ($attendance->clock_in_at) {
            $attendance->clock_in_at = Carbon::parse($attendance->clock_in_at)->format('Y-m-d H:i:s');
        }
        if ($attendance->clock_out_at) {
            $attendance->clock_out_at = Carbon::parse($attendance->clock_out_at)->format('Y-m-d H:i:s');
        }
        
        return response()->json($attendance);
    }

    /**
     * Manual attendance entry (for admins)
     */
    public function manualEntry(Request $request)
    {
        $data = $request->validate([
            'employee_id'   => 'required|exists:employees,id',
            'work_date'     => 'required|date',
            'clock_in_at'   => 'required|date_format:Y-m-d H:i:s',
            'clock_out_at'  => 'nullable|date_format:Y-m-d H:i:s|after:clock_in_at',
            'break_minutes' => 'nullable|integer|min:0',
            'notes'         => 'nullable|string',
            'source'        => 'nullable|string',
        ]);

        // Check if attendance already exists for this date
        $existing = EmployeeAttendance::where('employee_id', $data['employee_id'])
            ->where('work_date', $data['work_date'])
            ->first();

        if ($existing) {
            return response()->json([
                'message' => 'Attendance already exists for this date. Use update instead.'
            ], 422);
        }

        $clockIn = Carbon::parse($data['clock_in_at']);
        $clockOut = isset($data['clock_out_at']) ? Carbon::parse($data['clock_out_at']) : null;
        
        $totalMinutes = 0;
        $status = 'open';
        
        if ($clockOut) {
            $minutes = $clockIn->diffInMinutes($clockOut);
            $breakMinutes = $data['break_minutes'] ?? 0;
            $totalMinutes = max(0, $minutes - $breakMinutes);
            $status = 'completed';
        }

        $attendance = EmployeeAttendance::create([
            'employee_id'   => $data['employee_id'],
            'work_date'     => $data['work_date'],
            'clock_in_at'   => $clockIn,
            'clock_out_at'  => $clockOut,
            'break_minutes' => $data['break_minutes'] ?? 0,
            'total_minutes' => $totalMinutes,
            'status'        => $status,
            'source'        => $data['source'] ?? 'manual',
            'notes'         => $data['notes'] ?? null,
        ]);

        return response()->json([
            'message'    => 'Attendance created successfully',
            'attendance' => $attendance,
        ], 201);
    }

    /**
     * Update attendance record
     */
    public function update(Request $request, $id)
    {
        $attendance = EmployeeAttendance::findOrFail($id);

        $data = $request->validate([
            'work_date'     => 'sometimes|date',
            'clock_in_at'   => 'sometimes|date_format:Y-m-d H:i:s',
            'clock_out_at'  => 'nullable|date_format:Y-m-d H:i:s',
            'break_minutes' => 'nullable|integer|min:0',
            'notes'         => 'nullable|string',
            'status'        => 'sometimes|in:open,completed',
        ]);

        // Update fields
        if (isset($data['work_date'])) {
            $attendance->work_date = $data['work_date'];
        }

        if (isset($data['clock_in_at'])) {
            $attendance->clock_in_at = Carbon::parse($data['clock_in_at']);
        }

        if (isset($data['clock_out_at'])) {
            $attendance->clock_out_at = $data['clock_out_at'] ? Carbon::parse($data['clock_out_at']) : null;
        }

        if (isset($data['break_minutes'])) {
            $attendance->break_minutes = $data['break_minutes'];
        }

        if (isset($data['notes'])) {
            $attendance->notes = $data['notes'];
        }

        if (isset($data['status'])) {
            $attendance->status = $data['status'];
        }

        // Recalculate total minutes if both clock in and out exist
        if ($attendance->clock_in_at && $attendance->clock_out_at) {
            $minutes = Carbon::parse($attendance->clock_in_at)->diffInMinutes(Carbon::parse($attendance->clock_out_at));
            $attendance->total_minutes = max(0, $minutes - ($attendance->break_minutes ?? 0));
            $attendance->status = 'completed';
        }

        $attendance->save();

        return response()->json([
            'message'    => 'Attendance updated successfully',
            'attendance' => $attendance,
        ]);
    }

    /**
     * Delete attendance record
     */
    public function destroy($id)
    {
        $attendance = EmployeeAttendance::findOrFail($id);
        $attendance->delete();

        return response()->json([
            'message' => 'Attendance deleted successfully'
        ]);
    }
}