<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\ContractTemplate;

class ContractTemplateController extends Controller
{
    /**
     * List all contract templates
     */
    public function index(Request $request)
    {
        $query = ContractTemplate::query();

        if ($request->has('active_only') && $request->active_only) {
            $query->where('is_active', true);
        }

        $templates = $query->orderByDesc('is_default')->orderByDesc('id')->get();

        return response()->json($templates);
    }

    /**
     * Get single template
     */
    public function show(ContractTemplate $template)
    {
        return response()->json($template);
    }

    /**
     * Create new template
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => 'required|string|max:255',
            'template_content' => 'required|string',
            'description' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        $template = ContractTemplate::create($data);

        return response()->json($template, 201);
    }

    /**
     * Update template
     */
    public function update(Request $request, ContractTemplate $template)
    {
        $data = $request->validate([
            'name' => 'sometimes|string|max:255',
            'template_content' => 'sometimes|string',
            'description' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        $template->update($data);

        return response()->json($template);
    }

    /**
     * Delete template
     */
    public function destroy(ContractTemplate $template)
    {
        // Prevent deleting default template
        if ($template->is_default) {
            return response()->json([
                'error' => 'Cannot delete default template. Set another template as default first.'
            ], 422);
        }

        $template->delete();

        return response()->json(['message' => 'Template deleted successfully']);
    }

    /**
     * Set template as default
     */
    public function setAsDefault(ContractTemplate $template)
    {
        // Remove default from all templates
        ContractTemplate::where('is_default', true)->update(['is_default' => false]);

        // Set this template as default and activate it
        $template->update([
            'is_default' => true,
            'is_active' => true,
        ]);

        return response()->json($template);
    }
}
