<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Sale;
use App\Models\Purchase;
use App\Models\Inventory;
use App\Models\Customer;
use App\Models\Supplier;
use App\Models\InventoryItem;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DashboardController extends Controller
{
    /**
     * Get dashboard statistics
     */
    public function stats(Request $request): JsonResponse
    {
        try {
            // Total Revenue (from completed sales)
            $totalRevenue = Sale::where('status', 'completed')
                ->sum('total_amount');

            // Total Sales count
            $totalSales = Sale::count();

            // Today's sales count
            $todaySales = Sale::whereDate('created_at', today())->count();

            // Total Inventory items
            $totalInventory = Inventory::count();

            // Total Customers
            $totalCustomers = Customer::count();

            // Active Customers (who made purchases in last 6 months)
            $activeCustomers = Customer::whereHas('sales', function ($query) {
                $query->where('created_at', '>=', now()->subMonths(6));
            })->count();

            // Total Purchases
            $totalPurchases = Purchase::count();

            // Draft Purchases
            $draftPurchases = Purchase::where('status', 'draft')->count();

            // Total Suppliers
            $totalSuppliers = Supplier::count();

            // Active Suppliers (used in last 6 months)
            $activeSuppliers = Supplier::whereHas('purchases', function ($query) {
                $query->where('created_at', '>=', now()->subMonths(6));
            })->count();

            // Low Stock Items (assuming alert_quantity field exists)
            $lowStockItems = Inventory::whereColumn('quantity', '<=', 'alert_quantity')
                ->orWhere('quantity', '<=', 5)
                ->count();

            // Pending Payments (sales with payment_status != 'paid')
            $pendingPayments = Sale::whereIn('payment_status', ['pending', 'partial'])
                ->count();

            // Available Devices (inventory items not sold)
            $availableDevices = InventoryItem::where('status', 'available')
                ->where('is_active', true)
                ->count();

            // Sold Devices
            $soldDevices = InventoryItem::where('status', 'sold')
                ->count();

            // Calculate trends (compare with last month)
            $lastMonthRevenue = Sale::where('status', 'completed')
                ->whereMonth('created_at', now()->subMonth()->month)
                ->sum('total_amount');

            $revenueTrend = $lastMonthRevenue > 0 
                ? (($totalRevenue - $lastMonthRevenue) / $lastMonthRevenue) * 100 
                : 0;

            $lastMonthSales = Sale::whereMonth('created_at', now()->subMonth()->month)->count();
            $salesTrend = $lastMonthSales > 0 
                ? (($totalSales - $lastMonthSales) / $lastMonthSales) * 100 
                : 0;

            return response()->json([
                'success' => true,
                'data' => [
                    'totalRevenue' => $totalRevenue,
                    'revenueTrend' => round($revenueTrend, 1),
                    'totalSales' => $totalSales,
                    'salesTrend' => round($salesTrend, 1),
                    'todaySales' => $todaySales,
                    'totalInventory' => $totalInventory,
                    'totalCustomers' => $totalCustomers,
                    'activeCustomers' => $activeCustomers,
                    'totalPurchases' => $totalPurchases,
                    'draftPurchases' => $draftPurchases,
                    'totalSuppliers' => $totalSuppliers,
                    'activeSuppliers' => $activeSuppliers,
                    'lowStockItems' => $lowStockItems,
                    'pendingPayments' => $pendingPayments,
                    'availableDevices' => $availableDevices,
                    'soldDevices' => $soldDevices,
                ],
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load dashboard stats: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get payment statistics
     */
    public function paymentStats(Request $request): JsonResponse
    {
        try {
            // Paid orders
            $paidOrders = Sale::where('payment_status', 'paid')->get();
            $paid = $paidOrders->sum('total_amount');
            $paidCount = $paidOrders->count();

            // Partial payment orders
            $partialOrders = Sale::where('payment_status', 'partial')->get();
            $partial = $partialOrders->sum('due_amount');
            $partialCount = $partialOrders->count();

            // Unpaid orders
            $unpaidOrders = Sale::where('payment_status', 'pending')->get();
            $unpaid = $unpaidOrders->sum('total_amount');
            $unpaidCount = $unpaidOrders->count();

            return response()->json([
                'success' => true,
                'data' => [
                    'paid' => $paid,
                    'paidCount' => $paidCount,
                    'partial' => $partial,
                    'partialCount' => $partialCount,
                    'unpaid' => $unpaid,
                    'unpaidCount' => $unpaidCount,
                ],
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load payment stats: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get revenue trend (last 7 days)
     */
    public function revenueTrend(Request $request): JsonResponse
    {
        try {
            $days = 7;
            $data = [];

            for ($i = $days - 1; $i >= 0; $i--) {
                $date = Carbon::now()->subDays($i);
                $revenue = Sale::whereDate('created_at', $date)
                    ->where('status', 'completed')
                    ->sum('total_amount');

                $data[] = [
                    'date' => $date->format('Y-m-d'),
                    'day' => $date->format('D'),
                    'revenue' => $revenue,
                ];
            }

            return response()->json([
                'success' => true,
                'data' => $data,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load revenue trend: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get top selling products
     */
    public function topProducts(Request $request): JsonResponse
    {
        try {
            $limit = $request->get('limit', 5);

            // Get top selling inventory items
            $topProducts = DB::table('sale_items')
                ->join('inventory', 'sale_items.inventory_id', '=', 'inventory.id')
                ->select(
                    'inventory.id',
                    'inventory.model as name',
                    'inventory.sku',
                    'inventory.category',
                    DB::raw('SUM(sale_items.quantity) as sales'),
                    DB::raw('SUM(sale_items.subtotal) as revenue')
                )
                ->groupBy('inventory.id', 'inventory.model', 'inventory.sku', 'inventory.category')
                ->orderBy('sales', 'desc')
                ->limit($limit)
                ->get();

            return response()->json([
                'success' => true,
                'data' => $topProducts,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load top products: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get recent activity
     */
    public function recentActivity(Request $request): JsonResponse
    {
        try {
            $limit = $request->get('limit', 10);
            $activities = [];

            // Get recent sales
            $recentSales = Sale::with('customer')
                ->latest()
                ->limit(5)
                ->get();

            foreach ($recentSales as $sale) {
                $activities[] = [
                    'type' => 'sale',
                    'title' => 'New sale completed',
                    'description' => 'Order #' . $sale->order_number . ' - ' . ($sale->customer->name ?? 'Walk-in Customer'),
                    'time' => $sale->created_at->diffForHumans(),
                    'amount' => $sale->total_amount,
                    'created_at' => $sale->created_at,
                ];
            }

            // Get recent purchases
            $recentPurchases = Purchase::with('supplier')
                ->latest()
                ->limit(3)
                ->get();

            foreach ($recentPurchases as $purchase) {
                $activities[] = [
                    'type' => 'purchase',
                    'title' => 'Purchase order received',
                    'description' => $purchase->purchase_number . ' from ' . ($purchase->supplier->name ?? 'Unknown'),
                    'time' => $purchase->created_at->diffForHumans(),
                    'amount' => $purchase->total_amount,
                    'created_at' => $purchase->created_at,
                ];
            }

            // Get recent customers
            $recentCustomers = Customer::latest()
                ->limit(2)
                ->get();

            foreach ($recentCustomers as $customer) {
                $activities[] = [
                    'type' => 'customer',
                    'title' => 'New customer registered',
                    'description' => $customer->name . ' joined',
                    'time' => $customer->created_at->diffForHumans(),
                    'created_at' => $customer->created_at,
                ];
            }

            // Sort by created_at and limit
            $activities = collect($activities)
                ->sortByDesc('created_at')
                ->take($limit)
                ->values()
                ->toArray();

            return response()->json([
                'success' => true,
                'data' => $activities,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load recent activity: ' . $e->getMessage(),
            ], 500);
        }
    }
}