<?php

namespace App\Http\Controllers\Api;

use App\Models\ExpenseCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Http\Controllers\Controller;

class ExpenseCategoryController extends Controller
{
    // Get all categories with pagination and search
    public function index(Request $request)
    {
        try {
            $perPage = $request->input('per_page', 10);
            $search = $request->input('search', '');

            $query = ExpenseCategory::query();

            if ($search) {
                $query->where('name', 'like', '%' . $search . '%')
                      ->orWhere('description', 'like', '%' . $search . '%');
            }

            $categories = $query->orderBy('created_at', 'desc')->paginate($perPage);

            return response()->json($categories);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to fetch expense categories',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    // Get only active categories (for dropdowns)
    public function activeCategories()
    {
        try {
            $categories = ExpenseCategory::where('is_active', true)
                ->orderBy('name', 'asc')
                ->get(['id', 'name']);

            return response()->json($categories);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to fetch active categories',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    // Create new category
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255|unique:expense_categories,name',
                'description' => 'nullable|string',
                'is_active' => 'boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'error' => 'Validation failed',
                    'messages' => $validator->errors()
                ], 422);
            }

            $category = ExpenseCategory::create($request->all());

            return response()->json([
                'message' => 'Expense category created successfully',
                'data' => $category
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to create expense category',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    // Get single category
    public function show($id)
    {
        try {
            $category = ExpenseCategory::findOrFail($id);
            return response()->json($category);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Expense category not found',
                'message' => $e->getMessage()
            ], 404);
        }
    }

    // Update category
    public function update(Request $request, $id)
    {
        try {
            $category = ExpenseCategory::findOrFail($id);

            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255|unique:expense_categories,name,' . $id,
                'description' => 'nullable|string',
                'is_active' => 'boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'error' => 'Validation failed',
                    'messages' => $validator->errors()
                ], 422);
            }

            $category->update($request->all());

            return response()->json([
                'message' => 'Expense category updated successfully',
                'data' => $category
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to update expense category',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    // Delete category
    public function destroy($id)
    {
        try {
            $category = ExpenseCategory::findOrFail($id);
            
            // Check if category has expenses
            if ($category->expenses()->count() > 0) {
                return response()->json([
                    'error' => 'Cannot delete category with existing expenses'
                ], 400);
            }

            $category->delete();

            return response()->json([
                'message' => 'Expense category deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to delete expense category',
                'message' => $e->getMessage()
            ], 500);
        }
    }
}