<?php

namespace App\Http\Controllers\Api;

use App\Models\Expense;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Auth;
use App\Http\Controllers\Controller;

class ExpenseController extends Controller
{
    // Get all expenses with pagination, search, and filters
    public function index(Request $request)
    {
        try {
            $perPage = $request->input('per_page', 10);
            $search = $request->input('search', '');
            $categoryId = $request->input('category_id');
            $startDate = $request->input('start_date');
            $endDate = $request->input('end_date');

            $query = Expense::with(['category', 'supplier', 'creator']);

            // Search
            if ($search) {
                $query->where(function($q) use ($search) {
                    $q->where('description', 'like', '%' . $search . '%')
                      ->orWhere('reference_number', 'like', '%' . $search . '%')
                      ->orWhereHas('category', function($q) use ($search) {
                          $q->where('name', 'like', '%' . $search . '%');
                      });
                });
            }

            // Filter by category
            if ($categoryId) {
                $query->where('expense_category_id', $categoryId);
            }

            // Filter by date range
            if ($startDate) {
                $query->whereDate('expense_date', '>=', $startDate);
            }
            if ($endDate) {
                $query->whereDate('expense_date', '<=', $endDate);
            }

            $expenses = $query->orderBy('expense_date', 'desc')
                             ->orderBy('created_at', 'desc')
                             ->paginate($perPage);

            return response()->json($expenses);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to fetch expenses',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    // Create new expense
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'expense_category_id' => 'required|exists:expense_categories,id',
                'expense_date' => 'required|date',
                'amount' => 'required|numeric|min:0',
                'payment_method' => 'nullable|string|max:255',
                'reference_number' => 'nullable|string|max:255',
                'description' => 'nullable|string',
                'paid_to_staff_id' => 'nullable|exists:staff,id',
                'paid_to_supplier_id' => 'nullable|exists:suppliers,id'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'error' => 'Validation failed',
                    'messages' => $validator->errors()
                ], 422);
            }

            $data = $request->all();
            $data['created_by'] = Auth::id();

            $expense = Expense::create($data);
            $expense->load(['category',  'supplier', 'creator']);

            return response()->json([
                'message' => 'Expense created successfully',
                'data' => $expense
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to create expense',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    // Get single expense
    public function show($id)
    {
        try {
            $expense = Expense::with(['category',  'supplier', 'creator'])
                ->findOrFail($id);
            return response()->json($expense);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Expense not found',
                'message' => $e->getMessage()
            ], 404);
        }
    }

    // Update expense
    public function update(Request $request, $id)
    {
        try {
            $expense = Expense::findOrFail($id);

            $validator = Validator::make($request->all(), [
                'expense_category_id' => 'required|exists:expense_categories,id',
                'expense_date' => 'required|date',
                'amount' => 'required|numeric|min:0',
                'payment_method' => 'nullable|string|max:255',
                'reference_number' => 'nullable|string|max:255',
                'description' => 'nullable|string',
                'paid_to_staff_id' => 'nullable|exists:staff,id',
                'paid_to_supplier_id' => 'nullable|exists:suppliers,id'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'error' => 'Validation failed',
                    'messages' => $validator->errors()
                ], 422);
            }

            $expense->update($request->all());
            $expense->load(['category',  'supplier', 'creator']);

            return response()->json([
                'message' => 'Expense updated successfully',
                'data' => $expense
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to update expense',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    // Delete expense
    public function destroy($id)
    {
        try {
            $expense = Expense::findOrFail($id);
            
            // Delete receipt file if exists
            if ($expense->receipt_file && Storage::exists($expense->receipt_file)) {
                Storage::delete($expense->receipt_file);
            }

            $expense->delete();

            return response()->json([
                'message' => 'Expense deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to delete expense',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    // Upload receipt
    public function uploadReceipt(Request $request, $id)
    {
        try {
            $expense = Expense::findOrFail($id);

            $validator = Validator::make($request->all(), [
                'receipt' => 'required|file|mimes:jpg,jpeg,png,pdf|max:5120' // 5MB max
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'error' => 'Validation failed',
                    'messages' => $validator->errors()
                ], 422);
            }

            // Delete old receipt if exists
            if ($expense->receipt_file && Storage::exists($expense->receipt_file)) {
                Storage::delete($expense->receipt_file);
            }

            // Store new receipt
            $file = $request->file('receipt');
            $path = $file->store('receipts', 'public');

            $expense->receipt_file = $path;
            $expense->save();

            return response()->json([
                'message' => 'Receipt uploaded successfully',
                'file_path' => $path
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to upload receipt',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    // Delete receipt
    public function deleteReceipt($id)
    {
        try {
            $expense = Expense::findOrFail($id);

            if ($expense->receipt_file && Storage::exists($expense->receipt_file)) {
                Storage::delete($expense->receipt_file);
            }

            $expense->receipt_file = null;
            $expense->save();

            return response()->json([
                'message' => 'Receipt deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to delete receipt',
                'message' => $e->getMessage()
            ], 500);
        }
    }
}