<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use App\Models\Order;
use App\Models\Installment;
use App\Models\InstallmentPayment;
use App\Models\ContractTemplate;
use Carbon\Carbon;
use Barryvdh\DomPDF\Facade\Pdf;

class InstallmentController extends Controller
{
    public function index(Request $request)
    {
        $query = Installment::with(['order.customer', 'order.items.inventoryItem']);

        if ($request->status) {
            $query->where('status', $request->status);
        }

        if ($request->customer_cnic) {
            $query->where('customer_cnic', 'like', '%' . $request->customer_cnic . '%');
        }

        $installments = $query->orderByDesc('id')->paginate(20);

        return response()->json($installments);
    }

    public function show(Installment $installment)
    {
        $installment->load([
            'order.customer',
            'order.items.inventoryItem',
            'documents',
            'payments'
        ]);

        return response()->json($installment);
    }

    /**
     * Create installment order - Accepts JSON with base64 files
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'order_date' => 'nullable|date',
            'notes' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.inventory_item_id' => 'required|exists:inventory_items,id',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.discount_amount' => 'nullable|numeric|min:0',
            'items.*.tax_amount' => 'nullable|numeric|min:0',
            'customer_cnic' => 'required|string|min:13',
            'customer_address' => 'nullable|string',
            'customer_contact' => 'nullable|string',
            'guarantor_name' => 'required|string|min:3',
            'guarantor_cnic' => 'required|string|min:13',
            'guarantor_contact' => 'required|string|min:10',
            'guarantor_address' => 'nullable|string',
            'guarantor_relation' => 'nullable|string',
            'down_payment' => 'required|numeric|min:0',
            'markup_percentage' => 'required|numeric|min:0|max:100',
            'number_of_installments' => 'required|integer|min:1',
            'installment_frequency' => 'required|in:daily,weekly,bi-weekly,monthly',
            'start_date' => 'required|date',
            'terms_and_conditions' => 'nullable|string',
            'documents' => 'nullable|array',
            'documents.*.type' => 'required|in:customer_cnic,guarantor_cnic,utility_bill,salary_slip,bank_statement,employment_letter,other',
            'documents.*.base64' => 'required|string',
            'documents.*.filename' => 'required|string',
            'documents.*.description' => 'nullable|string',
        ]);

        try {
            $result = DB::transaction(function () use ($validated) {
                
                // Create order
                $orderNumber = 'IO-' . now()->format('Ymd-His') . '-' . \Str::random(4);

                $order = Order::create([
                    'order_number' => $orderNumber,
                    'order_type' => 'installment',
                    'customer_id' => $validated['customer_id'],
                    'order_date' => $validated['order_date'] ?? now()->toDateString(),
                    'status' => 'confirmed',
                    'payment_status' => 'partial',
                    'notes' => $validated['notes'] ?? null,
                ]);

                // Add items
                $subTotal = 0;
                $discountTotal = 0;
                $taxTotal = 0;

                foreach ($validated['items'] as $itemData) {
                    $device = \App\Models\InventoryItem::lockForUpdate()
                        ->findOrFail($itemData['inventory_item_id']);

                    if ($device->status !== 'available') {
                        throw new \Exception("Device IMEI {$device->imei_1} is not available.");
                    }

                    $lineDiscount = $itemData['discount_amount'] ?? 0;
                    $lineTax = $itemData['tax_amount'] ?? 0;
                    $unitPrice = $itemData['unit_price'];
                    $lineTotal = $unitPrice - $lineDiscount + $lineTax;

                    $order->items()->create([
                        'inventory_id' => $device->inventory_id,
                        'inventory_item_id' => $device->id,
                        'description' => $device->inventory->model . ' ' . ($device->storage ?? ''),
                        'qty' => 1,
                        'unit_price' => $unitPrice,
                        'discount_amount' => $lineDiscount,
                        'tax_amount' => $lineTax,
                        'total' => $lineTotal,
                    ]);

                    $device->update([
                        'status' => 'reserved',
                        'customer_id' => $order->customer_id,
                    ]);

                    $subTotal += $unitPrice;
                    $discountTotal += $lineDiscount;
                    $taxTotal += $lineTax;
                }

                $grandTotal = $subTotal - $discountTotal + $taxTotal;

                $order->update([
                    'sub_total' => $subTotal,
                    'discount_total' => $discountTotal,
                    'tax_total' => $taxTotal,
                    'grand_total' => $grandTotal,
                    'amount_paid' => $validated['down_payment'],
                ]);

                // Calculate installment
                $financedAmount = $grandTotal - $validated['down_payment'];
                $markupAmount = ($financedAmount * $validated['markup_percentage']) / 100;
                $totalPayable = $financedAmount + $markupAmount;
                $installmentAmount = $totalPayable / $validated['number_of_installments'];

                // Create installment
                $installment = $order->installment()->create([
                    'customer_cnic' => $validated['customer_cnic'],
                    'customer_address' => $validated['customer_address'] ?? null,
                    'customer_contact' => $validated['customer_contact'] ?? null,
                    'guarantor_name' => $validated['guarantor_name'],
                    'guarantor_cnic' => $validated['guarantor_cnic'],
                    'guarantor_contact' => $validated['guarantor_contact'],
                    'guarantor_address' => $validated['guarantor_address'] ?? null,
                    'guarantor_relation' => $validated['guarantor_relation'] ?? null,
                    'total_amount' => $grandTotal,
                    'down_payment' => $validated['down_payment'],
                    'financed_amount' => $financedAmount,
                    'markup_percentage' => $validated['markup_percentage'],
                    'markup_amount' => $markupAmount,
                    'total_payable' => $totalPayable,
                    'number_of_installments' => $validated['number_of_installments'],
                    'installment_amount' => $installmentAmount,
                    'installment_frequency' => $validated['installment_frequency'],
                    'start_date' => $validated['start_date'],
                    'status' => 'active',
                    'terms_and_conditions' => $validated['terms_and_conditions'] ?? null,
                ]);

                // Generate payment schedule
                $this->generatePaymentSchedule($installment);

                // Handle base64 documents
                if (!empty($validated['documents'])) {
                    foreach ($validated['documents'] as $doc) {
                        // Decode base64
                        $fileData = base64_decode($doc['base64']);
                        
                        // Generate filename
                        $fileName = time() . '_' . uniqid() . '_' . $doc['filename'];
                        $filePath = 'installment_documents/' . $fileName;
                        
                        // Save file
                        Storage::disk('public')->put($filePath, $fileData);
                        
                        // Get file size and mime type
                        $fullPath = storage_path('app/public/' . $filePath);
                        $fileSize = filesize($fullPath);
                        $mimeType = mime_content_type($fullPath);
                        
                        // Create document record
                        $installment->documents()->create([
                            'document_type' => $doc['type'],
                            'file_name' => $fileName,
                            'file_path' => $filePath,
                            'mime_type' => $mimeType,
                            'file_size' => $fileSize,
                            'description' => $doc['description'] ?? null,
                        ]);
                    }
                }

                return [
                    'order' => $order->fresh(['items.inventoryItem', 'customer']),
                    'installment' => $installment->fresh(['documents', 'payments']),
                ];
            });

            return response()->json([
                'message' => 'Installment order created successfully',
                'data' => $result
            ], 201);

        } catch (\Exception $e) {
            \Log::error('Installment creation failed: ' . $e->getMessage());
            
            return response()->json([
                'message' => 'Failed to create installment',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    private function generatePaymentSchedule(Installment $installment)
    {
        $startDate = Carbon::parse($installment->start_date);
        
        for ($i = 1; $i <= $installment->number_of_installments; $i++) {
            $dueDate = clone $startDate;

            switch ($installment->installment_frequency) {
                case 'daily':
                    $dueDate->addDays($i - 1);
                    break;
                case 'weekly':
                    $dueDate->addWeeks($i - 1);
                    break;
                case 'bi-weekly':
                    $dueDate->addWeeks(($i - 1) * 2);
                    break;
                case 'monthly':
                default:
                    $dueDate->addMonths($i - 1);
                    break;
            }

            InstallmentPayment::create([
                'installment_id' => $installment->id,
                'installment_number' => $i,
                'due_date' => $dueDate,
                'amount_due' => $installment->installment_amount,
                'status' => 'pending',
            ]);
        }

        $lastPayment = $installment->payments()->latest('due_date')->first();
        $installment->update(['end_date' => $lastPayment->due_date]);
    }

    public function recordPayment(Request $request, InstallmentPayment $payment)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:0.01',
            'payment_method' => 'nullable|string',
            'reference' => 'nullable|string',
            'notes' => 'nullable|string',
        ]);

        DB::transaction(function () use ($payment, $validated) {
            $totalDue = $payment->getTotalDue();
            $amountPaid = $payment->amount_paid + $validated['amount'];

            $payment->update([
                'amount_paid' => $amountPaid,
                'paid_date' => now(),
                'status' => $amountPaid >= $totalDue ? 'paid' : 'partial',
                'payment_method' => $validated['payment_method'] ?? null,
                'reference' => $validated['reference'] ?? null,
                'notes' => $validated['notes'] ?? null,
            ]);

            $installment = $payment->installment;
            if ($installment->isCompleted()) {
                $installment->update(['status' => 'completed']);

                $order = $installment->order;
                $order->update([
                    'status' => 'completed',
                    'payment_status' => 'paid',
                    'amount_paid' => $installment->getTotalPaidAmount(),
                ]);

                foreach ($order->items as $item) {
                    $device = $item->inventoryItem;
                    if ($device && $device->status === 'reserved') {
                        $device->update([
                            'status' => 'sold',
                            'sold_date' => now(),
                        ]);
                    }
                }
            } else {
                $installment->order->update([
                    'amount_paid' => $installment->getTotalPaidAmount(),
                ]);
            }
        });

        return response()->json($payment->fresh());
    }

    public function getPaymentSchedule(Installment $installment)
    {
        return response()->json($installment->payments()->orderBy('installment_number')->get());
    }

    public function overdueInstallments()
    {
        $gracePeriod = 3;
        $overdueDate = Carbon::now()->subDays($gracePeriod);

        $payments = InstallmentPayment::with(['installment.order.customer'])
            ->whereIn('status', ['pending', 'partial'])
            ->where('due_date', '<', $overdueDate)
            ->orderBy('due_date')
            ->paginate(20);

        foreach ($payments as $payment) {
            if ($payment->status !== 'overdue') {
                $payment->update([
                    'status' => 'overdue',
                    'late_fee' => $payment->calculateLateFee(),
                ]);
            }
        }

        return response()->json($payments);
    }

    public function upcomingPayments()
    {
        $payments = InstallmentPayment::with(['installment.order.customer'])
            ->where('status', 'pending')
            ->whereBetween('due_date', [now(), now()->addDays(7)])
            ->orderBy('due_date')
            ->paginate(20);

        return response()->json($payments);
    }

    public function statistics()
    {
        return response()->json([
            'total_active' => Installment::where('status', 'active')->count(),
            'total_completed' => Installment::where('status', 'completed')->count(),
            'total_defaulted' => Installment::where('status', 'defaulted')->count(),
            'total_amount_financed' => Installment::where('status', 'active')->sum('financed_amount'),
            'total_amount_collected' => InstallmentPayment::where('status', 'paid')->sum('amount_paid'),
            'overdue_count' => InstallmentPayment::where('status', 'overdue')->count(),
            'overdue_amount' => InstallmentPayment::where('status', 'overdue')->sum('amount_due'),
        ]);
    }

    public function generateContract(Installment $installment)
    {
        $installment->load(['order.customer', 'order.items.inventoryItem']);

        $template = ContractTemplate::getDefault();
        if (!$template) {
            return response()->json(['error' => 'No contract template found'], 404);
        }

        $deviceDetails = '';
        foreach ($installment->order->items as $item) {
            $device = $item->inventoryItem;
            $deviceDetails .= "<p>{$item->description} - IMEI: {$device->imei_1} - Rs. " . number_format($item->unit_price, 2) . "</p>";
        }

        $data = [
            'contract_date' => now()->format('d-m-Y'),
            'order_number' => $installment->order->order_number,
            'customer_name' => $installment->order->customer->name ?? 'N/A',
            'customer_cnic' => $installment->customer_cnic,
            'customer_contact' => $installment->customer_contact ?? 'N/A',
            'customer_address' => $installment->customer_address ?? 'N/A',
            'guarantor_name' => $installment->guarantor_name,
            'guarantor_cnic' => $installment->guarantor_cnic,
            'guarantor_contact' => $installment->guarantor_contact,
            'guarantor_address' => $installment->guarantor_address ?? 'N/A',
            'guarantor_relation' => $installment->guarantor_relation ?? 'N/A',
            'device_details' => $deviceDetails,
            'total_amount' => number_format($installment->total_amount, 2),
            'down_payment' => number_format($installment->down_payment, 2),
            'financed_amount' => number_format($installment->financed_amount, 2),
            'markup_percentage' => $installment->markup_percentage,
            'markup_amount' => number_format($installment->markup_amount, 2),
            'total_payable' => number_format($installment->total_payable, 2),
            'number_of_installments' => $installment->number_of_installments,
            'installment_amount' => number_format($installment->installment_amount, 2),
            'installment_frequency' => ucfirst($installment->installment_frequency),
            'start_date' => $installment->start_date->format('d-m-Y'),
        ];

        $html = $template->render($data);
        $pdf = Pdf::loadHTML($html);
        
        $fileName = 'contract_' . $installment->order->order_number . '_' . time() . '.pdf';
        $filePath = 'contracts/' . $fileName;
        Storage::disk('public')->put($filePath, $pdf->output());

        $installment->update([
            'contract_generated_at' => now(),
            'contract_path' => $filePath,
        ]);

        return response()->json([
            'message' => 'Contract generated',
            'contract_url' => asset('storage/' . $filePath),
        ]);
    }

    public function updateStatus(Request $request, Installment $installment)
    {
        $validated = $request->validate([
            'status' => 'required|in:active,completed,defaulted,suspended,cancelled',
            'notes' => 'nullable|string',
        ]);

        $installment->update($validated);

        if (in_array($validated['status'], ['defaulted', 'cancelled'])) {
            foreach ($installment->order->items as $item) {
                $device = $item->inventoryItem;
                if ($device && $device->status === 'reserved') {
                    $device->update(['status' => 'available', 'customer_id' => null]);
                }
            }
            $installment->order->update(['status' => 'cancelled']);
        }

        return response()->json($installment->fresh());
    }
}