<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Inventory;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Log;

class InventoryController extends Controller
{
    /**
     * Display a listing of inventory
     */
    public function index(Request $request): JsonResponse
    {
        $query = Inventory::with(['brand', 'category', 'supplier'])
            ->withCount([
                'items as total_items',
                'items as available_items' => function($q) {
                    $q->where('status', 'available');
                },
                'items as sold_items' => function($q) {
                    $q->where('status', 'sold');
                },
                'items as reserved_items' => function($q) {
                    $q->where('status', 'reserved');
                }
            ]);

        // Filter by brand
        if ($request->has('brand_id')) {
            $query->where('brand_id', $request->brand_id);
        }

        // Filter by category
        if ($request->has('category_id')) {
            $query->where('category_id', $request->category_id);
        }

        // Filter by status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Filter by condition
        if ($request->has('condition')) {
            $query->where('condition', $request->condition);
        }

        // Search by SKU, model, or brand
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('sku', 'like', "%{$search}%")
                  ->orWhere('model', 'like', "%{$search}%")
                  ->orWhereHas('brand', function($q) use ($search) {
                      $q->where('name', 'like', "%{$search}%");
                  });
            });
        }

        $inventory = $query->paginate(15);
        
        return response()->json([
            'success' => true,
            'data' => $inventory,
        ]);
    }

    /**
     * Store a newly created inventory item (product catalog entry)
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'sku' => 'required|string|unique:inventory',
            'brand_id' => 'required|exists:brands,id',
            'category_id' => 'required|exists:categories,id',
            'model' => 'required|string|max:255',
            
            // Default values for devices
            'color' => 'nullable|string|max:100',
            'storage' => 'nullable|string|max:100',
            'ram' => 'nullable|string|max:100',
            
            // Base pricing
            'purchase_price' => 'required|numeric|min:0',
            'selling_price' => 'required|numeric|min:0',
            'wholesale_price' => 'nullable|numeric|min:0',
            
            'minimum_stock' => 'nullable|integer|min:0',
            'description' => 'nullable|string',
            'condition' => 'required|in:new,refurbished,used',
            'warranty_period' => 'nullable|string',
            
            // Specifications
            'camera' => 'nullable|string',
            'battery' => 'nullable|string',
            'screen_size' => 'nullable|string',
            'processor' => 'nullable|string',
            'operating_system' => 'nullable|string',
            
            'supplier_id' => 'nullable|exists:suppliers,id',
            'status' => 'sometimes|in:available,out_of_stock,discontinued',
            'is_active' => 'sometimes|boolean',
            'image' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $data = $request->all();
        
        // Set initial status
        $data['status'] = $data['status'] ?? 'available';

        $inventory = Inventory::create($data);

        return response()->json([
            'success' => true,
            'message' => 'Product created successfully. You can now add individual devices.',
            'data' => $inventory->load(['brand', 'category', 'supplier']),
        ], 201);
    }

    /**
     * Display the specified inventory item with device counts
     */
    public function show($id): JsonResponse
    {
        $inventory = Inventory::with(['brand', 'category', 'supplier'])
            ->withCount([
                'items as total_items',
                'items as available_items' => function($q) {
                    $q->where('status', 'available');
                },
                'items as sold_items' => function($q) {
                    $q->where('status', 'sold');
                },
                'items as reserved_items' => function($q) {
                    $q->where('status', 'reserved');
                }
            ])
            ->find($id);

        if (!$inventory) {
            return response()->json([
                'success' => false,
                'message' => 'Product not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $inventory,
        ]);
    }

    /**
     * Update the specified inventory item
     */
    public function update(Request $request, $id): JsonResponse
    {
        $inventory = Inventory::find($id);

        if (!$inventory) {
            return response()->json([
                'success' => false,
                'message' => 'Product not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'sku' => ['sometimes', 'string', Rule::unique('inventory')->ignore($inventory->id)],
            'brand_id' => 'sometimes|exists:brands,id',
            'category_id' => 'sometimes|exists:categories,id',
            'model' => 'sometimes|string|max:255',
            'color' => 'nullable|string|max:100',
            'storage' => 'nullable|string|max:100',
            'ram' => 'nullable|string|max:100',
            'purchase_price' => 'sometimes|numeric|min:0',
            'selling_price' => 'sometimes|numeric|min:0',
            'wholesale_price' => 'nullable|numeric|min:0',
            'minimum_stock' => 'nullable|integer|min:0',
            'description' => 'nullable|string',
            'condition' => 'sometimes|in:new,refurbished,used',
            'warranty_period' => 'nullable|string',
            'camera' => 'nullable|string',
            'battery' => 'nullable|string',
            'screen_size' => 'nullable|string',
            'processor' => 'nullable|string',
            'operating_system' => 'nullable|string',
            'supplier_id' => 'nullable|exists:suppliers,id',
            'status' => 'sometimes|in:available,out_of_stock,discontinued',
            'is_active' => 'sometimes|boolean',
            'image' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $inventory->update($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Product updated successfully',
            'data' => $inventory->load(['brand', 'category', 'supplier']),
        ]);
    }

    /**
     * Remove the specified inventory item
     */
    public function destroy($id): JsonResponse
    {
        $inventory = Inventory::find($id);

        if (!$inventory) {
            return response()->json([
                'success' => false,
                'message' => 'Product not found'
            ], 404);
        }

        // Check if there are any devices
        if ($inventory->items()->exists()) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete product with existing devices. Please delete all devices first.'
            ], 422);
        }

        // Check if item has transactions
        if ($inventory->purchaseItems()->exists() || $inventory->saleItems()->exists()) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete product with existing transactions'
            ], 422);
        }

        $inventory->delete();

        return response()->json([
            'success' => true,
            'message' => 'Product deleted successfully',
        ]);
    }

    /**
     * Get low stock items (based on available devices count)
     */
    public function lowStock(): JsonResponse
    {
        $inventory = Inventory::with(['brand', 'category'])
            ->withCount([
                'items as available_items' => function($q) {
                    $q->where('status', 'available');
                }
            ])
            ->get()
            ->filter(function($item) {
                return $item->available_items > 0 && 
                       $item->available_items <= $item->minimum_stock;
            })
            ->values();
        
        return response()->json([
            'success' => true,
            'data' => $inventory,
        ]);
    }

    /**
     * Get out of stock items (no available devices)
     */
    public function outOfStock(): JsonResponse
    {
        $inventory = Inventory::with(['brand', 'category'])
            ->withCount([
                'items as available_items' => function($q) {
                    $q->where('status', 'available');
                }
            ])
            ->get()
            ->filter(function($item) {
                return $item->available_items === 0;
            })
            ->values();
        
        return response()->json([
            'success' => true,
            'data' => $inventory,
        ]);
    }

    /**
     * Get available items (products with available devices)
     */
    public function available(): JsonResponse
    {
        $inventory = Inventory::with(['brand', 'category'])
            ->where('status', 'available')
            ->where('is_active', true)
            ->withCount([
                'items as available_items' => function($q) {
                    $q->where('status', 'available');
                }
            ])
            ->get()
            ->filter(function($item) {
                return $item->available_items > 0;
            })
            ->values();
        
        return response()->json([
            'success' => true,
            'data' => $inventory,
        ]);
    }

    /**
     * Get inventory statistics
     */
    public function statistics(): JsonResponse
    {
        $stats = [
            'total_products' => Inventory::count(),
            'active_products' => Inventory::where('is_active', true)->count(),
            'total_devices' => \App\Models\InventoryItem::count(),
            'available_devices' => \App\Models\InventoryItem::where('status', 'available')->count(),
            'sold_devices' => \App\Models\InventoryItem::where('status', 'sold')->count(),
            'reserved_devices' => \App\Models\InventoryItem::where('status', 'reserved')->count(),
            'low_stock_products' => Inventory::all()->filter(function($item) {
                return $item->isLowStock();
            })->count(),
            'out_of_stock_products' => Inventory::all()->filter(function($item) {
                return $item->isOutOfStock();
            })->count(),
            'total_inventory_value' => \App\Models\InventoryItem::sum('purchase_price'),
            'available_inventory_value' => \App\Models\InventoryItem::where('status', 'available')->sum('purchase_price'),
            'total_brands' => \App\Models\Brand::count(),
            'total_categories' => \App\Models\Category::count(),
        ];

        return response()->json([
            'success' => true,
            'data' => $stats,
        ]);
    }

    /**
     * Check SKU availability
     */
    public function checkSku(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'sku' => 'required|string',
            'exclude_id' => 'nullable|integer',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $query = Inventory::where('sku', $request->sku);
        
        if ($request->has('exclude_id')) {
            $query->where('id', '!=', $request->exclude_id);
        }

        $exists = $query->exists();

        return response()->json([
            'available' => !$exists,
            'message' => $exists ? 'SKU already exists' : 'SKU is available'
        ]);
    }
}