<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\InventoryItem;
use App\Models\Inventory;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Carbon\Carbon;

class InventoryItemController extends Controller
{
    /**
     * Get all items for a specific inventory product
     */
    public function index(Request $request, $inventoryId = null): JsonResponse
    {
        $query = InventoryItem::with(['inventory.brand', 'inventory.category', 'supplier', 'customer']);

        // Filter by inventory (product)
        if ($inventoryId) {
            $query->where('inventory_id', $inventoryId);
        }

        // Filter by status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Filter by condition
        if ($request->has('condition')) {
            $query->where('condition', $request->condition);
        }

        // Search by IMEI or serial number
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('imei_1', 'like', "%{$search}%")
                    ->orWhere('imei_2', 'like', "%{$search}%")
                    ->orWhere('serial_number', 'like', "%{$search}%");
            });
        }

        // Filter by color
        if ($request->has('color')) {
            $query->where('color', $request->color);
        }

        // Filter by storage
        if ($request->has('storage')) {
            $query->where('storage', $request->storage);
        }

        // Sort by created_at descending (newest first)
        $query->orderBy('created_at', 'desc');

        // Get per_page from request, default to 1000 for large datasets
        // Allow 'all' to get all records without pagination
        $perPage = $request->input('per_page', 1000);

        if ($perPage === 'all' || $perPage > 10000) {
            // Get all records without pagination
            $items = $query->get();
            
            return response()->json([
                'success' => true,
                'data' => [
                    'data' => $items,
                    'total' => $items->count(),
                    'per_page' => $items->count(),
                    'current_page' => 1,
                    'last_page' => 1,
                ],
            ]);
        }

        // Paginate results
        $items = $query->paginate($perPage);

        return response()->json([
            'success' => true,
            'data' => $items,
        ]);
    }

    /**
     * Store a new inventory item (single device)
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'inventory_id' => 'required|exists:inventory,id',
            'imei_1' => 'required|string|size:15|unique:inventory_items,imei_1',
            'imei_2' => 'nullable|string|size:15|unique:inventory_items,imei_2',
            'serial_number' => 'nullable|string',
            'color' => 'nullable|string|max:100',
            'storage' => 'nullable|string|max:100',
            'ram' => 'nullable|string|max:100',
            'purchase_price' => 'required|numeric|min:0',
            'selling_price' => 'required|numeric|min:0',
            'supplier_id' => 'nullable|exists:suppliers,id',
            'purchase_date' => 'nullable|date',
            'condition' => 'required|in:new,refurbished,used',
            'warranty_months' => 'nullable|integer|min:0',
            'location' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
            'status' => 'sometimes|in:available,sold,reserved,damaged,in_repair,returned',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $data = $request->all();

        // Calculate warranty expiry if warranty_months is provided
        if ($request->warranty_months && $request->purchase_date) {
            $data['warranty_expiry_date'] = Carbon::parse($request->purchase_date)
                ->addMonths($request->warranty_months);
        }

        $item = InventoryItem::create($data);

        // Update parent inventory quantity
        $this->updateInventoryQuantity($item->inventory_id);

        return response()->json([
            'success' => true,
            'message' => 'Device added successfully',
            'data' => $item->load(['inventory.brand', 'inventory.category', 'supplier']),
        ], 201);
    }

    /**
     * Bulk add inventory items
     */
    public function bulkStore(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'inventory_id' => 'required|exists:inventory,id',
            'items' => 'required|array|min:1',
            'items.*.imei_1' => 'required|string|size:15|distinct|unique:inventory_items,imei_1',
            'items.*.imei_2' => 'nullable|string|size:15|distinct|unique:inventory_items,imei_2',
            'items.*.serial_number' => 'nullable|string',
            'items.*.color' => 'nullable|string|max:100',
            'items.*.storage' => 'nullable|string|max:100',
            'items.*.ram' => 'nullable|string|max:100',
            'items.*.purchase_price' => 'required|numeric|min:0',
            'items.*.selling_price' => 'required|numeric|min:0',
            'items.*.condition' => 'required|in:new,refurbished,used',
            'items.*.location' => 'nullable|string|max:255',
            'items.*.notes' => 'nullable|string',
            'supplier_id' => 'nullable|exists:suppliers,id',
            'purchase_date' => 'nullable|date',
            'warranty_months' => 'nullable|integer|min:0',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $createdItems = [];
        $warrantyExpiry = null;

        if ($request->warranty_months && $request->purchase_date) {
            $warrantyExpiry = Carbon::parse($request->purchase_date)
                ->addMonths($request->warranty_months);
        }

        foreach ($request->items as $itemData) {
            $itemData['inventory_id'] = $request->inventory_id;
            $itemData['supplier_id'] = $request->supplier_id;
            $itemData['purchase_date'] = $request->purchase_date;
            $itemData['warranty_months'] = $request->warranty_months;
            $itemData['warranty_expiry_date'] = $warrantyExpiry;
            $itemData['status'] = $itemData['status'] ?? 'available';

            $createdItems[] = InventoryItem::create($itemData);
        }

        // Update parent inventory quantity
        $this->updateInventoryQuantity($request->inventory_id);

        return response()->json([
            'success' => true,
            'message' => count($createdItems) . ' devices added successfully',
            'data' => $createdItems,
        ], 201);
    }

    /**
     * Display a specific inventory item by IMEI or ID
     */
    public function show($identifier): JsonResponse
    {
        // Try to find by IMEI first, then by ID
        $item = InventoryItem::where('imei_1', $identifier)
            ->orWhere('imei_2', $identifier)
            ->orWhere('id', $identifier)
            ->with(['inventory.brand', 'inventory.category', 'supplier', 'customer'])
            ->first();

        if (!$item) {
            return response()->json([
                'success' => false,
                'message' => 'Device not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $item,
        ]);
    }

    /**
     * Update an inventory item
     */
    public function update(Request $request, $id): JsonResponse
    {
        $item = InventoryItem::find($id);

        if (!$item) {
            return response()->json([
                'success' => false,
                'message' => 'Device not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'imei_1' => ['sometimes', 'string', 'size:15', Rule::unique('inventory_items')->ignore($item->id)],
            'imei_2' => ['nullable', 'string', 'size:15', Rule::unique('inventory_items')->ignore($item->id)],
            'serial_number' => 'nullable|string',
            'color' => 'nullable|string|max:100',
            'storage' => 'nullable|string|max:100',
            'ram' => 'nullable|string|max:100',
            'purchase_price' => 'sometimes|numeric|min:0',
            'selling_price' => 'sometimes|numeric|min:0',
            'supplier_id' => 'nullable|exists:suppliers,id',
            'purchase_date' => 'nullable|date',
            'condition' => 'sometimes|in:new,refurbished,used',
            'warranty_months' => 'nullable|integer|min:0',
            'location' => 'nullable|string|max:255',
            'status' => 'sometimes|in:available,sold,reserved,damaged,in_repair,returned',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $oldInventoryId = $item->inventory_id;
        $item->update($request->all());

        // Update warranty expiry if warranty_months changed
        if ($request->has('warranty_months') && $item->purchase_date) {
            $item->warranty_expiry_date = Carbon::parse($item->purchase_date)
                ->addMonths($request->warranty_months);
            $item->save();
        }

        // Update parent inventory quantities
        $this->updateInventoryQuantity($oldInventoryId);
        if ($request->has('inventory_id') && $request->inventory_id != $oldInventoryId) {
            $this->updateInventoryQuantity($request->inventory_id);
        }

        return response()->json([
            'success' => true,
            'message' => 'Device updated successfully',
            'data' => $item->load(['inventory.brand', 'inventory.category', 'supplier']),
        ]);
    }

    /**
     * Mark item as sold
     */
    public function markAsSold(Request $request, $id): JsonResponse
    {
        $item = InventoryItem::find($id);

        if (!$item) {
            return response()->json([
                'success' => false,
                'message' => 'Device not found'
            ], 404);
        }

        if ($item->status === 'sold') {
            return response()->json([
                'success' => false,
                'message' => 'Device already sold'
            ], 422);
        }

        $validator = Validator::make($request->all(), [
            'customer_id' => 'nullable|exists:customers,id',
            'sale_id' => 'nullable|exists:sales,id',
            'selling_price' => 'nullable|numeric|min:0',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $item->status = 'sold';
        $item->sold_date = now();

        if ($request->has('customer_id')) {
            $item->customer_id = $request->customer_id;
        }

        if ($request->has('sale_id')) {
            $item->sale_id = $request->sale_id;
        }

        if ($request->has('selling_price')) {
            $item->selling_price = $request->selling_price;
        }

        $item->save();

        // Update parent inventory quantity
        $this->updateInventoryQuantity($item->inventory_id);

        return response()->json([
            'success' => true,
            'message' => 'Device marked as sold',
            'data' => $item,
        ]);
    }

    /**
     * Delete an inventory item
     */
    public function destroy($id): JsonResponse
    {
        $item = InventoryItem::find($id);

        if (!$item) {
            return response()->json([
                'success' => false,
                'message' => 'Device not found'
            ], 404);
        }

        if ($item->status === 'sold') {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete sold device'
            ], 422);
        }

        $inventoryId = $item->inventory_id;
        $item->delete();

        // Update parent inventory quantity
        $this->updateInventoryQuantity($inventoryId);

        return response()->json([
            'success' => true,
            'message' => 'Device deleted successfully',
        ]);
    }

    /**
     * Get available items for a product
     */
    public function available(Request $request, $inventoryId): JsonResponse
    {
        $query = InventoryItem::where('inventory_id', $inventoryId)
            ->where('status', 'available')
            ->with(['supplier'])
            ->orderBy('created_at', 'desc');

        $perPage = $request->input('per_page', 1000);

        if ($perPage === 'all') {
            $items = $query->get();
            return response()->json([
                'success' => true,
                'data' => $items,
            ]);
        }

        $items = $query->paginate($perPage);

        return response()->json([
            'success' => true,
            'data' => $items,
        ]);
    }

    /**
     * Get sold items for a product
     */
    public function sold(Request $request, $inventoryId): JsonResponse
    {
        $query = InventoryItem::where('inventory_id', $inventoryId)
            ->where('status', 'sold')
            ->with(['customer', 'sale'])
            ->orderBy('sold_date', 'desc');

        $perPage = $request->input('per_page', 1000);

        if ($perPage === 'all') {
            $items = $query->get();
            return response()->json([
                'success' => true,
                'data' => $items,
            ]);
        }

        $items = $query->paginate($perPage);

        return response()->json([
            'success' => true,
            'data' => $items,
        ]);
    }

    /**
     * Search by IMEI
     */
    public function searchByImei(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'imei' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $item = InventoryItem::where('imei_1', 'like', '%' . $request->imei . '%')
            ->orWhere('imei_2', 'like', '%' . $request->imei . '%')
            ->with(['inventory.brand', 'inventory.category', 'supplier', 'customer'])
            ->first();

        if (!$item) {
            return response()->json([
                'success' => false,
                'message' => 'Device not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $item,
        ]);
    }

    /**
     * Helper method to update inventory quantity
     */
    private function updateInventoryQuantity($inventoryId): void
    {
        $inventory = Inventory::find($inventoryId);

        if ($inventory) {
            $availableCount = InventoryItem::where('inventory_id', $inventoryId)
                ->where('status', 'available')
                ->count();

            // Update quantity (total available devices)
            $inventory->quantity = $availableCount;

            // Update status
            if ($availableCount == 0) {
                $inventory->status = 'out_of_stock';
            } elseif ($inventory->status == 'out_of_stock') {
                $inventory->status = 'available';
            }

            $inventory->save();
        }
    }
}
