<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Inventory;
use App\Models\InventoryItem;
use App\Models\Accessory;
use App\Models\AccessoryStockMovement;

class OrderController extends Controller
{
    public function index(Request $request)
    {
        $orders = Order::with('customer')
            ->when($request->status, fn($q, $status) => $q->where('status', $status))
            ->orderByDesc('id')
            ->paginate(20);

        return response()->json($orders);
    }

    public function show(Order $order)
    {
        $order->load([
            'customer', 
            'items.inventoryItem.inventory',
            'items.accessory',
            'payments'
        ]);

        if ($order->order_type === 'installment') {
            $order->load([
                'installment.documents',
                'installment.payments' => function($query) {
                    $query->orderBy('installment_number');
                }
            ]);
        }

        return response()->json($order);
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'customer_id' => 'nullable|exists:customers,id',
            'order_date'  => 'nullable|date',
            'due_date'    => 'nullable|date|after_or_equal:order_date',
            'notes'       => 'nullable|string',
            'payment_status' => 'required|in:unpaid,partial,paid,refunded',
            'items'       => 'required|array|min:1',
            'items.*.item_type' => 'required|in:device,accessory',
            
            // Device fields
            'items.*.inventory_item_id' => 'required_if:items.*.item_type,device|exists:inventory_items,id',
            
            // Accessory fields
            'items.*.accessory_id' => 'required_if:items.*.item_type,accessory|exists:accessories,id',
            'items.*.qty' => 'required_if:items.*.item_type,accessory|integer|min:1',
            
            // Common fields
            'items.*.unit_price'        => 'required|numeric|min:0',
            'items.*.discount_amount'   => 'nullable|numeric|min:0',
            'items.*.tax_amount'        => 'nullable|numeric|min:0',
        ]);

        $order = DB::transaction(function () use ($data) {
            $orderNumber = 'SO-' . now()->format('Ymd-His') . '-' . Str::random(4);

            $order = Order::create([
                'order_number'   => $orderNumber,
                'order_type'     => 'cash',
                'customer_id'    => $data['customer_id'] ?? null,
                'order_date'     => $data['order_date'] ?? now()->toDateString(),
                'due_date'       => $data['due_date'] ?? null,
                'status'         => 'confirmed',
                'payment_status' => $data['payment_status'],
                'notes'          => $data['notes'] ?? null,
            ]);

            $subTotal = 0;
            $discountTotal = 0;
            $taxTotal = 0;

            foreach ($data['items'] as $itemData) {
                $lineDiscount = $itemData['discount_amount'] ?? 0;
                $lineTax      = $itemData['tax_amount'] ?? 0;
                $unitPrice    = $itemData['unit_price'];
                $payment_status = $data['payment_status'];

                if ($itemData['item_type'] === 'device') {
                    // Handle device
                    $device = InventoryItem::lockForUpdate()->findOrFail($itemData['inventory_item_id']);

                    if ($device->status !== 'available') {
                        throw new \Exception("Device IMEI {$device->imei_1} is not available.");
                    }

                    $qty = 1;
                    $lineTotal = ($unitPrice * $qty) - $lineDiscount + $lineTax;

                    $order->items()->create([
                        'item_type'         => 'device',
                        'inventory_id'      => $device->inventory_id,
                        'inventory_item_id' => $device->id,
                        'accessory_id'      => null,
                        'description'       => $device->inventory->model . ' ' . ($device->storage ?? ''),
                        'qty'               => $qty,
                        'unit_price'        => $unitPrice,
                        'discount_amount'   => $lineDiscount,
                        'tax_amount'        => $lineTax,
                        'total'             => $lineTotal,
                    ]);

                    $device->update([
                        'status'     => $payment_status == 'paid' ? 'sold' : 'reserved',
                        'customer_id' => $order->customer_id,
                    ]);

                    $subTotal      += ($unitPrice * $qty);
                    $discountTotal += $lineDiscount;
                    $taxTotal      += $lineTax;

                } elseif ($itemData['item_type'] === 'accessory') {
                    // Handle accessory
                    $accessory = Accessory::lockForUpdate()->findOrFail($itemData['accessory_id']);
                    $qty = $itemData['qty'];

                    if (!$accessory->is_active) {
                        throw new \Exception("Accessory {$accessory->name} is not active.");
                    }

                    if ($accessory->current_stock < $qty) {
                        throw new \Exception("Insufficient stock for {$accessory->name}. Available: {$accessory->current_stock}");
                    }

                    $lineTotal = ($unitPrice * $qty) - $lineDiscount + $lineTax;

                    $order->items()->create([
                        'item_type'         => 'accessory',
                        'inventory_id'      => null,
                        'inventory_item_id' => null,
                        'accessory_id'      => $accessory->id,
                        'description'       => $accessory->name . ' (' . $accessory->brand . ' ' . $accessory->model . ')',
                        'qty'               => $qty,
                        'unit_price'        => $unitPrice,
                        'discount_amount'   => $lineDiscount,
                        'tax_amount'        => $lineTax,
                        'total'             => $lineTotal,
                    ]);

                    // Reduce stock
                    $stockBefore = $accessory->current_stock;
                    $stockAfter = $stockBefore - $qty;
                    
                    $accessory->update([
                        'current_stock' => $stockAfter
                    ]);

                    // Create stock movement
                    AccessoryStockMovement::create([
                        'accessory_id' => $accessory->id,
                        'type' => 'sale',
                        'quantity' => -$qty,
                        'stock_before' => $stockBefore,
                        'stock_after' => $stockAfter,
                        'reference_id' => $order->id,
                        'reference_type' => Order::class,
                        'notes' => "Sale: Order #{$order->order_number}"
                    ]);

                    $subTotal      += ($unitPrice * $qty);
                    $discountTotal += $lineDiscount;
                    $taxTotal      += $lineTax;
                }
            }

            $grandTotal = $subTotal - $discountTotal + $taxTotal;

            $order->update([
                'sub_total'      => $subTotal,
                'discount_total' => $discountTotal,
                'tax_total'      => $taxTotal,
                'grand_total'    => $grandTotal,
                'amount_paid'    => $payment_status == 'paid' ? $grandTotal : 0,
            ]);

            return $order->fresh(['items.inventoryItem', 'items.accessory']);
        });

        return response()->json($order, 201);
    }

    public function addPayment(Request $request, Order $order)
    {
        $data = $request->validate([
            'amount'         => 'required|numeric|min:0.01',
            'payment_method' => 'nullable|string',
            'paid_at'        => 'nullable|date',
            'reference'      => 'nullable|string',
            'notes'          => 'nullable|string',
        ]);

        $payment = $order->payments()->create([
            'amount'         => $data['amount'],
            'payment_method' => $data['payment_method'] ?? null,
            'paid_at'        => $data['paid_at'] ?? now(),
            'reference'      => $data['reference'] ?? null,
            'notes'          => $data['notes'] ?? null,
        ]);

        $order->amount_paid += $payment->amount;
        if ($order->amount_paid >= $order->grand_total) {
            $order->payment_status = 'paid';
        } elseif ($order->amount_paid > 0) {
            $order->payment_status = 'partial';
        }
        $order->save();

        return response()->json($order->fresh(['payments']));
    }

    public function complete(Order $order)
    {
        if ($order->status === 'completed') {
            return response()->json(['message' => 'Order already completed'], 200);
        }

        DB::transaction(function () use ($order) {
            foreach ($order->items as $item) {
                if ($item->isDevice()) {
                    $device = InventoryItem::lockForUpdate()->find($item->inventory_item_id);
                    if (!$device) continue;

                    $device->update([
                        'status'     => 'sold',
                        'sold_date'  => now(),
                        'customer_id' => $order->customer_id,
                    ]);
                }
            }

            $order->status = 'completed';
            $order->save();
        });

        return response()->json($order->fresh(['items.inventoryItem', 'items.accessory']));
    }

    public function cancel(Order $order)
    {
        if (in_array($order->status, ['completed', 'cancelled'])) {
            return response()->json(['message' => 'Order cannot be cancelled.'], 422);
        }

        DB::transaction(function () use ($order) {
            if ($order->order_type === 'installment' && $order->installment) {
                $order->installment->update(['status' => 'cancelled']);
            }

            foreach ($order->items as $item) {
                if ($item->isDevice()) {
                    $device = InventoryItem::lockForUpdate()->find($item->inventory_item_id);
                    if ($device && in_array($device->status, ['reserved', 'sold'])) {
                        $device->update([
                            'status'      => 'available',
                            'customer_id' => null,
                        ]);
                    }
                } elseif ($item->isAccessory()) {
                    // Return stock
                    $accessory = Accessory::lockForUpdate()->find($item->accessory_id);
                    if ($accessory) {
                        $stockBefore = $accessory->current_stock;
                        $stockAfter = $stockBefore + $item->qty;
                        
                        $accessory->update([
                            'current_stock' => $stockAfter
                        ]);

                        AccessoryStockMovement::create([
                            'accessory_id' => $accessory->id,
                            'type' => 'return',
                            'quantity' => $item->qty,
                            'stock_before' => $stockBefore,
                            'stock_after' => $stockAfter,
                            'reference_id' => $order->id,
                            'reference_type' => Order::class,
                            'notes' => "Order cancelled: #{$order->order_number}"
                        ]);
                    }
                }
            }

            $order->status = 'cancelled';
            $order->save();
        });

        return response()->json($order->fresh(['items']));
    }
}