<?php

namespace App\Http\Controllers\Api;

use App\Models\Payroll;
use App\Models\Employee;
use App\Models\EmployeeAttendance;
use App\Models\EmployeeCommission;
use Illuminate\Http\Request;
use Carbon\Carbon;
use App\Http\Controllers\Controller;

class PayrollController extends Controller
{
    /**
     * List all payrolls
     */
    public function index(Request $request)
    {
        $perPage = $request->get('per_page', 15);
        
        $payrolls = Payroll::with('employee')
            ->orderBy('created_at', 'desc')
            ->paginate($perPage);

        return response()->json($payrolls);
    }

    /**
     * Show single payroll
     */
    public function show($id)
    {
        $payroll = Payroll::with('employee')->findOrFail($id);
        return response()->json($payroll);
    }

    /**
     * Generate new payroll
     */
    public function generate(Request $request)
    {
        // $data = $request->validate([
        //     'employee_id' => 'required|exists:employees,id',
        //     'period_start' => 'required|date',
        //     'period_end' => 'required|date|after_or_equal:period_start',
        //     'bonus' => 'nullable|numeric|min:0',
        //     'overtime_hours' => 'nullable|numeric|min:0',
        //     'deductions' => 'nullable|numeric|min:0',
        //     'notes' => 'nullable|string',
        // ]);

        $data = $request->validate([
    'employee_id' => 'required|exists:employees,id',
    'period_start' => 'required|date',
    'period_end' => 'required|date|after_or_equal:period_start',
    'bonus' => 'nullable|numeric|min:0',
    'overtime_hours' => 'nullable|numeric|min:0',
    'notes' => 'nullable|string',
    
    // Allowances
    'allowance_hra' => 'nullable|numeric|min:0',
    'allowance_fuel' => 'nullable|numeric|min:0',
    'allowance_phone' => 'nullable|numeric|min:0',
    'allowance_medical' => 'nullable|numeric|min:0',
    'allowance_meal' => 'nullable|numeric|min:0',
    'allowance_other' => 'nullable|numeric|min:0',
    
    // Deductions
    'deduction_tax' => 'nullable|numeric|min:0',
    'deduction_advance' => 'nullable|numeric|min:0',
    'deduction_loan' => 'nullable|numeric|min:0',
    'deduction_penalty' => 'nullable|numeric|min:0',
    'deduction_other' => 'nullable|numeric|min:0',
]);

        $employee = Employee::findOrFail($data['employee_id']);

        // Check if payroll already exists for this period
        $existing = Payroll::where('employee_id', $data['employee_id'])
            ->where('period_start', $data['period_start'])
            ->where('period_end', $data['period_end'])
            ->first();

        if ($existing) {
            return response()->json([
                'message' => 'Payroll already exists for this period'
            ], 422);
        }

        // Get attendance for the period
        $attendances = EmployeeAttendance::where('employee_id', $data['employee_id'])
            ->whereDate('work_date', '>=', $data['period_start'])
            ->whereDate('work_date', '<=', $data['period_end'])
            ->where('status', 'completed')
            ->get();

        $totalDays = $attendances->count();
        $totalMinutes = $attendances->sum('total_minutes');
        $totalHours = round($totalMinutes / 60, 2);

        // Calculate base pay based on salary type
        $basePay = 0;
        
        if ($employee->salary_type === 'fixed') {
            $basePay = $employee->base_salary ?? 0;
        } elseif ($employee->salary_type === 'daily') {
            $basePay = ($employee->daily_rate ?? 0) * $totalDays;
        } elseif ($employee->salary_type === 'hourly') {
            $basePay = ($employee->hourly_rate ?? 0) * $totalHours;
        }

        // Calculate commission if any
        $commissionPay = 0;
        if ($employee->commission_rate > 0) {
            $commissions = EmployeeCommission::where('employee_id', $data['employee_id'])
                ->whereDate('sale_date', '>=', $data['period_start'])
                ->whereDate('sale_date', '<=', $data['period_end'])
                ->get();
            
            $commissionPay = $commissions->sum('commission_amount');
        }

        // Calculate overtime pay
        $overtimeHours = $data['overtime_hours'] ?? 0;
        $hourlyRate = $employee->hourly_rate ?? 0;
        $overtimePay = $overtimeHours * $hourlyRate * 1.5;

        // Get bonus and deductions
        $bonus = $data['bonus'] ?? 0;
        $deductions = $data['deductions'] ?? 0;

        // Calculate net pay
        // $netPay = $basePay + $commissionPay + $overtimePay + $bonus - $deductions;


        // Get allowances (use employee defaults if not provided)
$allowanceHRA = $data['allowance_hra'] ?? $employee->allowance_hra ?? 0;
$allowanceFuel = $data['allowance_fuel'] ?? $employee->allowance_fuel ?? 0;
$allowancePhone = $data['allowance_phone'] ?? $employee->allowance_phone ?? 0;
$allowanceMedical = $data['allowance_medical'] ?? $employee->allowance_medical ?? 0;
$allowanceMeal = $data['allowance_meal'] ?? 0;
$allowanceOther = $data['allowance_other'] ?? 0;

$allowancesTotal = $allowanceHRA + $allowanceFuel + $allowancePhone 
                 + $allowanceMedical + $allowanceMeal + $allowanceOther;

// Calculate gross earnings
$grossEarnings = $basePay + $commissionPay + $overtimePay + $bonus + $allowancesTotal;

// Get deductions
$deductionTax = $data['deduction_tax'] ?? 0;
$deductionAdvance = $data['deduction_advance'] ?? 0;
$deductionLoan = $data['deduction_loan'] ?? 0;
$deductionPenalty = $data['deduction_penalty'] ?? 0;
$deductionOther = $data['deduction_other'] ?? 0;

$deductionsTotal = $deductionTax + $deductionAdvance + $deductionLoan 
                 + $deductionPenalty + $deductionOther;

// Calculate net pay
$netPay = $grossEarnings - $deductionsTotal;

        // Create payroll record
        // $payroll = Payroll::create([
        //     'employee_id' => $data['employee_id'],
        //     'period_start' => $data['period_start'],
        //     'period_end' => $data['period_end'],
        //     'total_days' => $totalDays,
        //     'total_hours' => $totalHours,
        //     'base_pay' => $basePay,
        //     'commission_pay' => $commissionPay,
        //     'overtime_hours' => $overtimeHours,
        //     'overtime_pay' => $overtimePay,
        //     'bonus' => $bonus,
        //     'deductions' => $deductions,
        //     'net_pay' => $netPay,
        //     'status' => 'draft',
        //     'notes' => $data['notes'] ?? null,
        //     'generated_by' => auth()->id() ?? null,
        // ]);

        $payroll = Payroll::create([
    'employee_id' => $data['employee_id'],
    'period_start' => $data['period_start'],
    'period_end' => $data['period_end'],
    'total_days' => $totalDays,
    'total_hours' => $totalHours,
    'base_pay' => $basePay,
    'commission_pay' => $commissionPay,
    'overtime_hours' => $overtimeHours,
    'overtime_pay' => $overtimePay,
    'bonus' => $bonus,
    
    // Allowances
    'allowance_hra' => $allowanceHRA,
    'allowance_fuel' => $allowanceFuel,
    'allowance_phone' => $allowancePhone,
    'allowance_medical' => $allowanceMedical,
    'allowance_meal' => $allowanceMeal,
    'allowance_other' => $allowanceOther,
    'allowances_total' => $allowancesTotal,
    
    // Gross
    'gross_earnings' => $grossEarnings,
    
    // Deductions
    'deduction_tax' => $deductionTax,
    'deduction_advance' => $deductionAdvance,
    'deduction_loan' => $deductionLoan,
    'deduction_penalty' => $deductionPenalty,
    'deduction_other' => $deductionOther,
    'deductions_total' => $deductionsTotal,
    
    'net_pay' => $netPay,
    'status' => 'draft',
    'notes' => $data['notes'] ?? null,
    'generated_by' => auth()->id() ?? null,
]);

        return response()->json([
            'message' => 'Payroll generated successfully',
            'payroll' => $payroll->load('employee'),
        ], 201);
    }

    /**
     * Update payroll status
     */
    public function updateStatus(Request $request, $id)
    {
        $payroll = Payroll::findOrFail($id);

        $data = $request->validate([
            'status' => 'required|in:draft,approved,paid',
            'payment_method' => 'nullable|string',
            'payment_reference' => 'nullable|string',
            'paid_at' => 'nullable|date',
        ]);

        // Status workflow validation
        if ($payroll->status === 'paid' && $data['status'] !== 'paid') {
            return response()->json([
                'message' => 'Cannot change status of paid payroll'
            ], 422);
        }

        if ($payroll->status === 'draft' && $data['status'] === 'paid') {
            return response()->json([
                'message' => 'Draft payroll must be approved before marking as paid'
            ], 422);
        }

        $payroll->status = $data['status'];

        if ($data['status'] === 'approved') {
            $payroll->approved_by = auth()->id() ?? null;
            $payroll->approved_at = now();
        }

        if ($data['status'] === 'paid') {
            $payroll->payment_method = $data['payment_method'] ?? null;
            $payroll->payment_reference = $data['payment_reference'] ?? null;
            $payroll->paid_at = $data['paid_at'] ?? now();
            $payroll->paid_by = auth()->id() ?? null;
        }

        $payroll->save();

        return response()->json([
            'message' => 'Payroll status updated successfully',
            'payroll' => $payroll->load('employee'),
        ]);
    }

    /**
     * Update payroll (draft only)
     */
    public function update(Request $request, $id)
    {
        $payroll = Payroll::findOrFail($id);

        // Only allow updates for draft payrolls
        if ($payroll->status !== 'draft') {
            return response()->json([
                'message' => 'Only draft payrolls can be updated'
            ], 422);
        }

        $data = $request->validate([
            'bonus' => 'nullable|numeric|min:0',
            'overtime_hours' => 'nullable|numeric|min:0',
            'deductions' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string',
        ]);

        // Update fields
        if (isset($data['bonus'])) {
            $payroll->bonus = $data['bonus'];
        }

        if (isset($data['overtime_hours'])) {
            $payroll->overtime_hours = $data['overtime_hours'];
            // Recalculate overtime pay
            $employee = $payroll->employee;
            $hourlyRate = $employee->hourly_rate ?? 0;
            $payroll->overtime_pay = $data['overtime_hours'] * $hourlyRate * 1.5;
        }

        if (isset($data['deductions'])) {
            $payroll->deductions = $data['deductions'];
        }

        if (isset($data['notes'])) {
            $payroll->notes = $data['notes'];
        }

        // Recalculate net pay
        $payroll->net_pay = $payroll->base_pay 
            + $payroll->commission_pay 
            + $payroll->overtime_pay 
            + $payroll->bonus 
            - $payroll->deductions;

        $payroll->save();

        return response()->json([
            'message' => 'Payroll updated successfully',
            'payroll' => $payroll->load('employee'),
        ]);
    }

    /**
     * Delete payroll (draft only)
     */
    public function destroy($id)
    {
        $payroll = Payroll::findOrFail($id);

        if ($payroll->status !== 'draft') {
            return response()->json([
                'message' => 'Only draft payrolls can be deleted'
            ], 422);
        }

        $payroll->delete();

        return response()->json([
            'message' => 'Payroll deleted successfully'
        ]);
    }
}