<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Purchase;
use App\Models\PurchaseItem;
use App\Models\PurchasePayment;
use App\Models\Inventory;
use App\Models\InventoryItem; // 👈 per-device table (IMEI)
use App\Models\Setting;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Log;

class PurchaseController extends Controller
{
    /**
     * Display a listing of purchases
     */
    public function index(Request $request): JsonResponse
    {
        $query = Purchase::with(['supplier', 'createdBy', 'purchaseItems.inventory']);

        // Filter by supplier
        if ($request->has('supplier_id')) {
            $query->where('supplier_id', $request->supplier_id);
        }

        // Filter by payment status
        if ($request->has('payment_status')) {
            $query->where('payment_status', $request->payment_status);
        }

        // Filter by status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Filter by date range
        if ($request->has('start_date') && $request->has('end_date')) {
            $query->whereBetween('purchase_date', [$request->start_date, $request->end_date]);
        }

        // Search by purchase number or reference
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('purchase_number', 'like', "%{$search}%")
                    ->orWhere('reference_number', 'like', "%{$search}%");
            });
        }

        $purchases = $query->orderBy('purchase_date', 'desc')->paginate(15);

        return response()->json([
            'success' => true,
            'data' => $purchases,
        ]);
    }

    /**
     * Store a newly created purchase
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'supplier_id'       => 'required|exists:suppliers,id',
            'purchase_date'     => 'required|date',
            'reference_number'  => 'nullable|string|max:255',
            'notes'             => 'nullable|string',
            'status'            => 'nullable|in:draft,completed',

            'items'                     => 'required|array|min:1',
            'items.*.inventory_id'      => 'required|exists:inventory,id',
            'items.*.quantity'          => 'required|integer|min:1',
            'items.*.unit_price'        => 'required|numeric|min:0',
            'items.*.is_device'         => 'nullable|boolean',

            'tax_amount'        => 'nullable|numeric|min:0',
            'discount'          => 'nullable|numeric|min:0',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors'  => $validator->errors(),
            ], 422);
        }

        DB::beginTransaction();

        try {
            // Generate purchase number
            $prefix       = Setting::get('purchase_prefix', 'PUR');
            $lastPurchase = Purchase::latest('id')->first();
            $number       = $lastPurchase ? $lastPurchase->id + 1 : 1;
            $purchaseNumber = $prefix . '-' . str_pad($number, 4, '0', STR_PAD_LEFT);

            // Calculate totals
            $subtotal = 0;
            foreach ($request->items as $item) {
                $subtotal += $item['quantity'] * $item['unit_price'];
            }

            $taxAmount   = $request->tax_amount ?? 0;
            $discount    = $request->discount ?? 0;
            $totalAmount = $subtotal + $taxAmount - $discount;

            // Get status from request, default to 'completed'
            $status = $request->status ?? 'completed';

            // Create purchase
            $purchase = Purchase::create([
                'purchase_number'  => $purchaseNumber,
                'supplier_id'      => $request->supplier_id,
                'created_by'       => auth()->id(),
                'purchase_date'    => $request->purchase_date,
                'subtotal'         => $subtotal,
                'tax_amount'       => $taxAmount,
                'discount'         => $discount,
                'total_amount'     => $totalAmount,
                'paid_amount'      => 0,
                'due_amount'       => $totalAmount,
                'payment_status'   => 'pending',
                'status'           => $status, // 👈 Use request status
                'reference_number' => $request->reference_number,
                'notes'            => $request->notes,
            ]);

            // Create purchase items, update inventory and create per-device inventory_items
            foreach ($request->items as $item) {
                $isDevice    = !empty($item['is_device']);
                $devicesData = $item['devices'] ?? [];
                $itemSubtotal = $item['quantity'] * $item['unit_price'];

                // Create purchase item record
                $purchaseItem = PurchaseItem::create([
                    'purchase_id'  => $purchase->id,
                    'inventory_id' => $item['inventory_id'],
                    'quantity'     => $item['quantity'],
                    'unit_price'   => $item['unit_price'],
                    'subtotal'     => $itemSubtotal,
                    'is_device'    => $isDevice ? 1 : 0,
                ]);

                // Load base inventory (parent product)
                /** @var Inventory $inventory */
                $inventory = Inventory::find($item['inventory_id']);

                // If this is a device-tracked item, create inventory_items rows (per device / per IMEI)
                if (count($devicesData) > 0) {
                    Log::info($devicesData);
                    foreach ($devicesData as $dev) {
                        Log::info($dev);

                        // For draft status, allow empty IMEI (user can fill it later)
                        if ($status === 'draft' && empty($dev['imei1'])) {
                            continue; // Skip device creation if draft and no IMEI
                        }

                        if (empty($dev['imei1'])) {
                            // For completed status, frontend should validate
                            continue;
                        }

                        InventoryItem::create([
                            'inventory_id'        => $inventory->id,
                            'purchase_item_id'    => $purchaseItem->id,
                            'imei_1'              => $dev['imei1'],
                            'imei_2'              => $dev['imei2'] ?? null,
                            'serial_number'       => $dev['serial_number'] ?? null,

                            'color'               => $dev['color'] ?? $inventory->color,
                            'storage'             => $dev['storage'] ?? $inventory->storage,
                            'ram'                 => $dev['ram'] ?? $inventory->ram,

                            'purchase_price'      => $dev['purchase_price'] ?? $item['unit_price'],
                            'selling_price'       => $dev['selling_price'] ?? ($inventory->selling_price ?? 0),

                            'supplier_id'         => $request->supplier_id,
                            'purchase_date'       => $request->purchase_date,

                            'condition'           => $dev['condition'] ?? 'new',
                            'warranty_months'     => $dev['warranty_months'] ?? null,
                            'warranty_expiry_date' => $dev['warranty_expiry_date'] ?? null,

                            'status'              => 'available', // in stock and ready
                            'location'            => $dev['location'] ?? null,

                            'sold_date'           => null,
                            'customer_id'         => null,

                            'notes'               => $dev['notes'] ?? null,
                            'is_active'           => true,
                        ]);
                    }
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => $status === 'draft'
                    ? 'Purchase saved as draft successfully'
                    : 'Purchase created successfully',
                'data'    => $purchase->load(['supplier', 'purchaseItems.inventory']),
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Failed to create purchase: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Display the specified purchase
     */
    public function show($id): JsonResponse
    {
        $purchase = Purchase::with([
            'supplier',
            'createdBy',
            'purchaseItems.inventory',
            'purchaseItems.inventoryItem', // 👈 Load devices for each item
            'purchasePayments.receivedBy',
        ])->find($id);

        if (!$purchase) {
            return response()->json([
                'success' => false,
                'message' => 'Purchase not found',
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data'    => $purchase,
        ]);
    }

    /**
     * Update the specified purchase
     */
    public function update(Request $request, $id): JsonResponse
    {
        $purchase = Purchase::find($id);

        if (!$purchase) {
            return response()->json([
                'success' => false,
                'message' => 'Purchase not found',
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'supplier_id'      => 'required|exists:suppliers,id',
            'purchase_date'    => 'required|date',
            'reference_number' => 'nullable|string|max:255',
            'notes'            => 'nullable|string',
            'status'           => 'nullable|in:draft,completed',

            'items'                    => 'required|array|min:1',
            'items.*.inventory_id'     => 'required|exists:inventory,id',
            'items.*.quantity'         => 'required|integer|min:1',
            'items.*.unit_price'       => 'required|numeric|min:0',

            'tax_amount'       => 'nullable|numeric|min:0',
            'discount'         => 'nullable|numeric|min:0',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors'  => $validator->errors(),
            ], 422);
        }

        DB::beginTransaction();

        try {
            // Delete old purchase items and their associated inventory items
            foreach ($purchase->purchaseItems as $oldItem) {
                // Delete associated inventory items (devices)
                InventoryItem::where('purchase_item_id', $oldItem->id)->delete();
                $oldItem->delete();
            }

            // Calculate new totals
            $subtotal = 0;
            foreach ($request->items as $item) {
                $subtotal += $item['quantity'] * $item['unit_price'];
            }

            $taxAmount   = $request->tax_amount ?? 0;
            $discount    = $request->discount ?? 0;
            $totalAmount = $subtotal + $taxAmount - $discount;

            // Get status from request, keep existing if not provided
            $status = $request->status ?? $purchase->status;

            // Update purchase
            $purchase->update([
                'supplier_id'      => $request->supplier_id,
                'purchase_date'    => $request->purchase_date,
                'subtotal'         => $subtotal,
                'tax_amount'       => $taxAmount,
                'discount'         => $discount,
                'total_amount'     => $totalAmount,
                'due_amount'       => $totalAmount - $purchase->paid_amount,
                'status'           => $status, // 👈 Update status
                'reference_number' => $request->reference_number,
                'notes'            => $request->notes,
            ]);

            // Create new purchase items with devices
            foreach ($request->items as $item) {
                $devicesData = $item['devices'] ?? [];
                $itemSubtotal = $item['quantity'] * $item['unit_price'];

                $purchaseItem = PurchaseItem::create([
                    'purchase_id'  => $purchase->id,
                    'inventory_id' => $item['inventory_id'],
                    'quantity'     => $item['quantity'],
                    'unit_price'   => $item['unit_price'],
                    'subtotal'     => $itemSubtotal,
                ]);

                $inventory = Inventory::find($item['inventory_id']);

                // Create inventory items (devices)
                if (count($devicesData) > 0) {
                    foreach ($devicesData as $dev) {
                        // For draft status, allow empty IMEI
                        if ($status === 'draft' && empty($dev['imei1'])) {
                            continue;
                        }

                        if (empty($dev['imei1'])) {
                            continue;
                        }

                        InventoryItem::create([
                            'inventory_id'        => $inventory->id,
                            'purchase_item_id'    => $purchaseItem->id,
                            'imei_1'              => $dev['imei1'],
                            'imei_2'              => $dev['imei2'] ?? null,
                            'serial_number'       => $dev['serial_number'] ?? null,

                            'color'               => $dev['color'] ?? $inventory->color,
                            'storage'             => $dev['storage'] ?? $inventory->storage,
                            'ram'                 => $dev['ram'] ?? $inventory->ram,

                            'purchase_price'      => $dev['purchase_price'] ?? $item['unit_price'],
                            'selling_price'       => $dev['selling_price'] ?? ($inventory->selling_price ?? 0),

                            'supplier_id'         => $request->supplier_id,
                            'purchase_date'       => $request->purchase_date,

                            'condition'           => $dev['condition'] ?? 'new',
                            'warranty_months'     => $dev['warranty_months'] ?? null,
                            'warranty_expiry_date' => $dev['warranty_expiry_date'] ?? null,

                            'status'              => 'available',
                            'location'            => $dev['location'] ?? null,

                            'sold_date'           => null,
                            'customer_id'         => null,

                            'notes'               => $dev['notes'] ?? null,
                            'is_active'           => true,
                        ]);
                    }
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => $status === 'draft'
                    ? 'Draft updated successfully'
                    : 'Purchase updated successfully',
                'data'    => $purchase->load(['supplier', 'purchaseItems.inventory']),
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Failed to update purchase: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Record a payment for a purchase
     */
    public function addPayment(Request $request, $id): JsonResponse
    {
        $purchase = Purchase::find($id);

        if (!$purchase) {
            return response()->json([
                'success' => false,
                'message' => 'Purchase not found',
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'amount'           => 'required|numeric|min:0.01',
            'payment_date'     => 'required|date',
            'payment_method'   => 'required|string|max:50',
            'reference_number' => 'nullable|string|max:255',
            'notes'            => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors'  => $validator->errors(),
            ], 422);
        }

        DB::beginTransaction();

        try {
            // Create payment record
            PurchasePayment::create([
                'purchase_id'      => $purchase->id,
                'amount'           => $request->amount,
                'payment_date'     => $request->payment_date,
                'payment_method'   => $request->payment_method,
                'reference_number' => $request->reference_number,
                'notes'            => $request->notes,
                'received_by'      => auth()->id(),
            ]);

            // Update purchase payment info
            $purchase->paid_amount += $request->amount;
            $purchase->due_amount   = $purchase->total_amount - $purchase->paid_amount;

            if ($purchase->due_amount <= 0) {
                $purchase->payment_status = 'paid';
                $purchase->due_amount = 0;
            } elseif ($purchase->paid_amount > 0) {
                $purchase->payment_status = 'partial';
            } else {
                $purchase->payment_status = 'pending';
            }

            $purchase->save();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Payment recorded successfully',
                'data'    => $purchase->load(['purchasePayments']),
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Failed to record payment: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Cancel a purchase
     */
    public function cancel($id): JsonResponse
    {
        $purchase = Purchase::with('purchaseItems')->find($id);

        if (!$purchase) {
            return response()->json([
                'success' => false,
                'message' => 'Purchase not found',
            ], 404);
        }

        if ($purchase->status === 'cancelled') {
            return response()->json([
                'success' => false,
                'message' => 'Purchase is already cancelled',
            ], 400);
        }

        DB::beginTransaction();

        try {
            // Delete all inventory items (devices) associated with this purchase
            foreach ($purchase->purchaseItems as $item) {
                InventoryItem::where('purchase_item_id', $item->id)->delete();
            }

            $purchase->status = 'cancelled';
            $purchase->save();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Purchase cancelled successfully',
                'data'    => $purchase,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Failed to cancel purchase: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get purchase statistics
     */
    public function stats(Request $request): JsonResponse
    {
        $query = Purchase::query();

        // Filter by date range
        if ($request->has('start_date') && $request->has('end_date')) {
            $query->whereBetween('purchase_date', [$request->start_date, $request->end_date]);
        }

        $totalPurchases = $query->count();
        $totalAmount    = $query->sum('total_amount');
        $totalPaid      = $query->sum('paid_amount');
        $totalDue       = $query->sum('due_amount');

        return response()->json([
            'success' => true,
            'data'    => [
                'total_purchases' => $totalPurchases,
                'total_amount'    => $totalAmount,
                'total_paid'      => $totalPaid,
                'total_due'       => $totalDue,
            ],
        ]);
    }
}
