<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Sale;
use App\Models\SaleItem;
use App\Models\SalePayment;
use App\Models\Installment;
use App\Models\Inventory;
use App\Models\Setting;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;

class SaleController extends Controller
{
    /**
     * Display a listing of sales
     */
    public function index(Request $request): JsonResponse
    {
        $query = Sale::with(['customer', 'soldBy', 'saleItems.inventory']);

        // Filter by customer
        if ($request->has('customer_id')) {
            $query->where('customer_id', $request->customer_id);
        }

        // Filter by payment status
        if ($request->has('payment_status')) {
            $query->where('payment_status', $request->payment_status);
        }

        // Filter by payment type
        if ($request->has('payment_type')) {
            $query->where('payment_type', $request->payment_type);
        }

        // Filter by status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Filter by date range
        if ($request->has('start_date') && $request->has('end_date')) {
            $query->whereBetween('sale_date', [$request->start_date, $request->end_date]);
        }

        // Search by invoice number
        if ($request->has('search')) {
            $search = $request->search;
            $query->where('invoice_number', 'like', "%{$search}%");
        }

        $sales = $query->orderBy('sale_date', 'desc')->paginate(15);
        
        return response()->json([
            'success' => true,
            'data' => $sales,
        ]);
    }

    /**
     * Store a newly created sale
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|exists:customers,id',
            'sale_date' => 'required|date',
            'payment_type' => 'required|in:cash,installment',
            'notes' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.inventory_id' => 'required|exists:inventory,id',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.unit_price' => 'required|numeric|min:0',
            'tax_amount' => 'nullable|numeric|min:0',
            'discount' => 'nullable|numeric|min:0',
            
            // Installment fields (required if payment_type is installment)
            'installment_count' => 'required_if:payment_type,installment|integer|min:1',
            'installment_frequency' => 'required_if:payment_type,installment|in:weekly,monthly',
            'down_payment' => 'nullable|numeric|min:0',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();
        try {
            // Check inventory availability
            foreach ($request->items as $item) {
                $inventory = Inventory::find($item['inventory_id']);
                if ($inventory->quantity < $item['quantity']) {
                    return response()->json([
                        'success' => false,
                        'message' => "Insufficient stock for {$inventory->model}. Available: {$inventory->quantity}"
                    ], 422);
                }
            }

            // Generate invoice number
            $prefix = Setting::get('invoice_prefix', 'INV');
            $lastSale = Sale::latest('id')->first();
            $number = $lastSale ? $lastSale->id + 1 : 1;
            $invoiceNumber = $prefix . '-' . str_pad($number, 4, '0', STR_PAD_LEFT);

            // Calculate totals and profit
            $subtotal = 0;
            $totalProfit = 0;

            foreach ($request->items as $item) {
                $inventory = Inventory::find($item['inventory_id']);
                $itemSubtotal = $item['quantity'] * $item['unit_price'];
                $itemProfit = ($item['unit_price'] - $inventory->purchase_price) * $item['quantity'];
                
                $subtotal += $itemSubtotal;
                $totalProfit += $itemProfit;
            }

            $taxAmount = $request->tax_amount ?? 0;
            $discount = $request->discount ?? 0;
            $totalAmount = $subtotal + $taxAmount - $discount;

            // For installment, calculate down payment
            $downPayment = 0;
            if ($request->payment_type == 'installment') {
                $downPayment = $request->down_payment ?? 0;
            }

            $paidAmount = $request->payment_type == 'cash' ? 0 : $downPayment;
            $dueAmount = $totalAmount - $paidAmount;

            // Create sale
            $sale = Sale::create([
                'invoice_number' => $invoiceNumber,
                'customer_id' => $request->customer_id,
                'sold_by' => auth()->id(),
                'sale_date' => $request->sale_date,
                'subtotal' => $subtotal,
                'tax_amount' => $taxAmount,
                'discount' => $discount,
                'total_amount' => $totalAmount,
                'paid_amount' => $paidAmount,
                'due_amount' => $dueAmount,
                'profit' => $totalProfit,
                'payment_status' => $paidAmount >= $totalAmount ? 'paid' : ($paidAmount > 0 ? 'partial' : 'pending'),
                'payment_type' => $request->payment_type,
                'status' => 'completed',
                'notes' => $request->notes,
            ]);

            // Create sale items and update inventory
            foreach ($request->items as $item) {
                $inventory = Inventory::find($item['inventory_id']);
                $itemSubtotal = $item['quantity'] * $item['unit_price'];
                $itemProfit = ($item['unit_price'] - $inventory->purchase_price) * $item['quantity'];
                
                SaleItem::create([
                    'sale_id' => $sale->id,
                    'inventory_id' => $item['inventory_id'],
                    'quantity' => $item['quantity'],
                    'unit_cost' => $inventory->purchase_price,
                    'unit_price' => $item['unit_price'],
                    'subtotal' => $itemSubtotal,
                    'profit' => $itemProfit,
                ]);

                // Update inventory quantity
                $inventory->quantity -= $item['quantity'];
                
                if ($inventory->quantity <= 0) {
                    $inventory->quantity = 0;
                    $inventory->status = 'out_of_stock';
                }
                
                $inventory->save();
            }

            // Record down payment if installment
            if ($request->payment_type == 'installment' && $downPayment > 0) {
                SalePayment::create([
                    'sale_id' => $sale->id,
                    'amount' => $downPayment,
                    'payment_date' => $request->sale_date,
                    'payment_method' => 'cash',
                    'notes' => 'Down payment',
                    'received_by' => auth()->id(),
                ]);
            }

            // Create installment plan if needed
            if ($request->payment_type == 'installment') {
                $remainingAmount = $totalAmount - $downPayment;
                $installmentAmount = $remainingAmount / $request->installment_count;
                
                $dueDate = Carbon::parse($request->sale_date);
                
                for ($i = 1; $i <= $request->installment_count; $i++) {
                    // Calculate next due date
                    if ($request->installment_frequency == 'weekly') {
                        $dueDate = $dueDate->addWeeks(1);
                    } else {
                        $dueDate = $dueDate->addMonth();
                    }
                    
                    Installment::create([
                        'sale_id' => $sale->id,
                        'installment_number' => $i,
                        'amount' => $installmentAmount,
                        'due_date' => $dueDate,
                        'paid_amount' => 0,
                        'status' => 'pending',
                    ]);
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Sale created successfully',
                'data' => $sale->load(['customer', 'saleItems.inventory', 'installments']),
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to create sale: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified sale
     */
    public function show($id): JsonResponse
    {
        $sale = Sale::with([
            'customer', 
            'soldBy', 
            'saleItems.inventory.brand',
            'salePayments.receivedBy',
            'installments'
        ])->find($id);

        if (!$sale) {
            return response()->json([
                'success' => false,
                'message' => 'Sale not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $sale,
        ]);
    }

    /**
     * Cancel sale
     */
    public function cancel($id): JsonResponse
    {
        $sale = Sale::find($id);

        if (!$sale) {
            return response()->json([
                'success' => false,
                'message' => 'Sale not found'
            ], 404);
        }

        if ($sale->status == 'cancelled') {
            return response()->json([
                'success' => false,
                'message' => 'Sale already cancelled'
            ], 422);
        }

        DB::beginTransaction();
        try {
            // Revert inventory quantities
            foreach ($sale->saleItems as $item) {
                $inventory = Inventory::find($item->inventory_id);
                $inventory->quantity += $item->quantity;
                
                if ($inventory->quantity > 0 && $inventory->status == 'out_of_stock') {
                    $inventory->status = 'available';
                }
                
                $inventory->save();
            }

            // Update sale status
            $sale->status = 'cancelled';
            $sale->save();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Sale cancelled successfully',
                'data' => $sale,
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to cancel sale: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Add payment to sale
     */
    public function addPayment(Request $request, $id): JsonResponse
    {
        $sale = Sale::find($id);

        if (!$sale) {
            return response()->json([
                'success' => false,
                'message' => 'Sale not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'amount' => 'required|numeric|min:0|max:' . $sale->due_amount,
            'payment_date' => 'required|date',
            'payment_method' => 'required|in:cash,bank_transfer,cheque,card,mobile_payment',
            'reference_number' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
            'installment_id' => 'nullable|exists:installments,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();
        try {
            // Create payment
            $payment = SalePayment::create([
                'sale_id' => $sale->id,
                'amount' => $request->amount,
                'payment_date' => $request->payment_date,
                'payment_method' => $request->payment_method,
                'reference_number' => $request->reference_number,
                'notes' => $request->notes,
                'received_by' => auth()->id(),
            ]);

            // Update sale payment status
            $sale->paid_amount += $request->amount;
            $sale->due_amount -= $request->amount;

            if ($sale->due_amount <= 0) {
                $sale->payment_status = 'paid';
            } elseif ($sale->paid_amount > 0) {
                $sale->payment_status = 'partial';
            }

            $sale->save();

            // Update installment if specified
            if ($request->has('installment_id')) {
                $installment = Installment::find($request->installment_id);
                if ($installment && $installment->sale_id == $sale->id) {
                    $installment->paid_amount += $request->amount;
                    
                    if ($installment->paid_amount >= $installment->amount) {
                        $installment->status = 'paid';
                        $installment->paid_date = $request->payment_date;
                    }
                    
                    $installment->save();
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Payment added successfully',
                'data' => [
                    'payment' => $payment,
                    'sale' => $sale,
                ],
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to add payment: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get sale payments
     */
    public function getPayments($id): JsonResponse
    {
        $sale = Sale::find($id);

        if (!$sale) {
            return response()->json([
                'success' => false,
                'message' => 'Sale not found'
            ], 404);
        }

        $payments = $sale->salePayments()->with('receivedBy')->get();

        return response()->json([
            'success' => true,
            'data' => $payments,
        ]);
    }

    /**
     * Get sale installments
     */
    public function getInstallments($id): JsonResponse
    {
        $sale = Sale::find($id);

        if (!$sale) {
            return response()->json([
                'success' => false,
                'message' => 'Sale not found'
            ], 404);
        }

        $installments = $sale->installments;

        return response()->json([
            'success' => true,
            'data' => $installments,
        ]);
    }

    /**
     * Pay specific installment
     */
    public function payInstallment(Request $request, $id): JsonResponse
    {
        $installment = Installment::find($id);

        if (!$installment) {
            return response()->json([
                'success' => false,
                'message' => 'Installment not found'
            ], 404);
        }

        if ($installment->status == 'paid') {
            return response()->json([
                'success' => false,
                'message' => 'Installment already paid'
            ], 422);
        }

        $validator = Validator::make($request->all(), [
            'amount' => 'required|numeric|min:0',
            'payment_date' => 'required|date',
            'payment_method' => 'required|in:cash,bank_transfer,cheque,card,mobile_payment',
            'reference_number' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();
        try {
            $sale = Sale::find($installment->sale_id);
            
            // Create payment
            $payment = SalePayment::create([
                'sale_id' => $sale->id,
                'amount' => $request->amount,
                'payment_date' => $request->payment_date,
                'payment_method' => $request->payment_method,
                'reference_number' => $request->reference_number,
                'notes' => $request->notes ?? "Payment for installment #{$installment->installment_number}",
                'received_by' => auth()->id(),
            ]);

            // Update installment
            $installment->paid_amount += $request->amount;
            
            if ($installment->paid_amount >= $installment->amount) {
                $installment->status = 'paid';
                $installment->paid_date = $request->payment_date;
            }
            
            $installment->save();

            // Update sale
            $sale->paid_amount += $request->amount;
            $sale->due_amount -= $request->amount;

            if ($sale->due_amount <= 0) {
                $sale->payment_status = 'paid';
            } elseif ($sale->paid_amount > 0) {
                $sale->payment_status = 'partial';
            }

            $sale->save();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Installment payment recorded successfully',
                'data' => [
                    'payment' => $payment,
                    'installment' => $installment,
                    'sale' => $sale,
                ],
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to record payment: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get sales with pending payments
     */
    public function pendingPayments(): JsonResponse
    {
        $sales = Sale::with(['customer', 'soldBy'])
                     ->whereIn('payment_status', ['pending', 'partial'])
                     ->where('status', 'completed')
                     ->orderBy('sale_date', 'desc')
                     ->paginate(15);

        return response()->json([
            'success' => true,
            'data' => $sales,
        ]);
    }

    /**
     * Get overdue installments
     */
    public function overdueInstallments(): JsonResponse
    {
        $installments = Installment::with(['sale.customer'])
                                   ->where('status', 'pending')
                                   ->where('due_date', '<', now())
                                   ->orderBy('due_date', 'asc')
                                   ->paginate(15);

        // Update status to overdue
        foreach ($installments as $installment) {
            if ($installment->status == 'pending') {
                $installment->status = 'overdue';
                $installment->save();
            }
        }

        return response()->json([
            'success' => true,
            'data' => $installments,
        ]);
    }

    /**
     * Get today's due installments
     */
    public function todayDueInstallments(): JsonResponse
    {
        $today = now()->toDateString();
        
        $installments = Installment::with(['sale.customer'])
                                   ->where('status', 'pending')
                                   ->whereDate('due_date', $today)
                                   ->get();

        return response()->json([
            'success' => true,
            'data' => $installments,
        ]);
    }

    /**
     * Get sale statistics
     */
    public function statistics(Request $request): JsonResponse
    {
        $query = Sale::where('status', 'completed');

        // Filter by date range
        if ($request->has('start_date') && $request->has('end_date')) {
            $query->whereBetween('sale_date', [$request->start_date, $request->end_date]);
        }

        $totalSales = $query->count();
        $totalRevenue = $query->sum('total_amount');
        $totalProfit = $query->sum('profit');
        $totalPaid = $query->sum('paid_amount');
        $totalDue = $query->sum('due_amount');
        
        $cashSales = Sale::where('status', 'completed')
                         ->where('payment_type', 'cash')
                         ->when($request->has('start_date') && $request->has('end_date'), function($q) use ($request) {
                             $q->whereBetween('sale_date', [$request->start_date, $request->end_date]);
                         })
                         ->count();
                         
        $installmentSales = Sale::where('status', 'completed')
                                ->where('payment_type', 'installment')
                                ->when($request->has('start_date') && $request->has('end_date'), function($q) use ($request) {
                                    $q->whereBetween('sale_date', [$request->start_date, $request->end_date]);
                                })
                                ->count();

        return response()->json([
            'success' => true,
            'data' => [
                'total_sales' => $totalSales,
                'total_revenue' => $totalRevenue,
                'total_profit' => $totalProfit,
                'total_paid' => $totalPaid,
                'total_due' => $totalDue,
                'cash_sales' => $cashSales,
                'installment_sales' => $installmentSales,
                'profit_margin' => $totalRevenue > 0 ? ($totalProfit / $totalRevenue) * 100 : 0,
            ],
        ]);
    }

    /**
     * Get top selling products
     */
    public function topSellingProducts(Request $request): JsonResponse
    {
        $limit = $request->limit ?? 10;
        
        $products = SaleItem::with(['inventory.brand', 'inventory.category'])
                            ->select('inventory_id', DB::raw('SUM(quantity) as total_sold'), DB::raw('SUM(subtotal) as total_revenue'))
                            ->when($request->has('start_date') && $request->has('end_date'), function($q) use ($request) {
                                $q->whereHas('sale', function($query) use ($request) {
                                    $query->whereBetween('sale_date', [$request->start_date, $request->end_date]);
                                });
                            })
                            ->groupBy('inventory_id')
                            ->orderBy('total_sold', 'desc')
                            ->limit($limit)
                            ->get();

        return response()->json([
            'success' => true,
            'data' => $products,
        ]);
    }

    /**
     * Get sales by staff member
     */
    public function salesByStaff(Request $request): JsonResponse
    {
        $query = Sale::with(['soldBy'])
                     ->where('status', 'completed')
                     ->select('sold_by', 
                             DB::raw('COUNT(*) as total_sales'), 
                             DB::raw('SUM(total_amount) as total_revenue'),
                             DB::raw('SUM(profit) as total_profit'));

        // Filter by date range
        if ($request->has('start_date') && $request->has('end_date')) {
            $query->whereBetween('sale_date', [$request->start_date, $request->end_date]);
        }

        $salesByStaff = $query->groupBy('sold_by')
                              ->orderBy('total_revenue', 'desc')
                              ->get();

        return response()->json([
            'success' => true,
            'data' => $salesByStaff,
        ]);
    }

    /**
     * Get daily sales report
     */
    public function dailySalesReport(Request $request): JsonResponse
    {
        $date = $request->date ?? now()->toDateString();
        
        $sales = Sale::with(['customer', 'soldBy', 'saleItems.inventory'])
                     ->whereDate('sale_date', $date)
                     ->where('status', 'completed')
                     ->get();
        
        $totalSales = $sales->count();
        $totalRevenue = $sales->sum('total_amount');
        $totalProfit = $sales->sum('profit');
        $totalCash = $sales->where('payment_type', 'cash')->sum('total_amount');
        $totalInstallment = $sales->where('payment_type', 'installment')->sum('total_amount');

        return response()->json([
            'success' => true,
            'data' => [
                'date' => $date,
                'sales' => $sales,
                'summary' => [
                    'total_sales' => $totalSales,
                    'total_revenue' => $totalRevenue,
                    'total_profit' => $totalProfit,
                    'cash_sales' => $totalCash,
                    'installment_sales' => $totalInstallment,
                ],
            ],
        ]);
    }
}