<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

class SearchController extends Controller
{
    /**
     * Generic search endpoint
     *
     * GET /api/search?model=customers&q=john&per_page=20
     */
    public function index(Request $request)
    {
        $validated = $request->validate([
            'model'    => 'required|string',
            'q'        => 'nullable|string',
            'per_page' => 'nullable|integer|min=1|max:100',
        ]);

        $modelKey = strtolower($validated['model']);
        $search   = $validated['q'] ?? null;
        $perPage  = $validated['per_page'] ?? 20;

        /**
         * Map allowed "modules" to Eloquent models + searchable fields
         * You can extend this as needed.
         */
        $modules = [
            'customers' => [
                'class'       => \App\Models\Customer::class,
                'searchable'  => ['name', 'first_name', 'last_name', 'email', 'phone'],
                'order_by'    => ['created_at', 'desc'],
            ],
            'suppliers' => [
                'class'       => \App\Models\Supplier::class,
                'searchable'  => ['name', 'contact_name', 'email', 'phone'],
                'order_by'    => ['created_at', 'desc'],
            ],
            'inventory' => [
                'class'       => \App\Models\Inventory::class,
                'searchable'  => ['model', 'sku', 'category'],
                'order_by'    => ['model', 'asc'],
            ],
            // add more modules here...
        ];

        // Block unknown modules (security)
        if (! array_key_exists($modelKey, $modules)) {
            return response()->json([
                'message' => 'Unsupported module: '.$modelKey,
            ], 404);
        }

        $config = $modules[$modelKey];

        /** @var \Illuminate\Database\Eloquent\Builder $query */
        $query = $config['class']::query();

        // Apply search if q is provided
        if ($search) {
            $query->where(function ($q) use ($config, $search) {
                foreach ($config['searchable'] as $field) {
                    $q->orWhere($field, 'LIKE', '%'.$search.'%');
                }
            });
        }

        // Default ordering
        if (! empty($config['order_by'])) {
            [$column, $direction] = $config['order_by'];
            $query->orderBy($column, $direction);
        }

        // Paginate
        $results = $query->get();

        // return response()->json($results);
        return response()->json([
            'model'   => $modelKey,
            'query'   => $search,
            'perPage' => (int) $perPage,
            'data'    => $results, // this is a LengthAwarePaginator (with links/meta)
        ]);
    }
}
