<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class StaffController extends Controller
{
    /**
     * Display a listing of staff members
     */
    public function index(): JsonResponse
    {
        // $staff = User::where('role', 'staff')->get();
        $staff = User::all();
        
        return response()->json([
            'success' => true,
            'data' => $staff,
        ]);
    }

    /**
     * Store a newly created staff member
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'username' => 'required|string|max:255|unique:users',
            'email' => 'required|string|email|max:255|unique:users',
            'phone' => 'required|string|max:20|unique:users',
            'password' => 'required|string|min:8',
            'is_active' => 'sometimes|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $staff = User::create([
            'name' => $request->name,
            'username' => $request->username,
            'email' => $request->email,
            'phone' => $request->phone,
            'password' => Hash::make($request->password),
            'role' => 'staff',
            'is_active' => $request->is_active ?? true,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Staff member created successfully',
            'data' => $staff,
        ], 201);
    }

    /**
     * Display the specified staff member
     */
    public function show($id): JsonResponse
    {
        $staff = User::where('role', 'staff')->find($id);

        if (!$staff) {
            return response()->json([
                'success' => false,
                'message' => 'Staff member not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $staff,
        ]);
    }

    /**
     * Update the specified staff member
     */
    public function update(Request $request, $id): JsonResponse
    {
        $staff = User::where('role', 'staff')->find($id);

        if (!$staff) {
            return response()->json([
                'success' => false,
                'message' => 'Staff member not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|string|max:255',
            'username' => ['sometimes', 'string', 'max:255', Rule::unique('users')->ignore($staff->id)],
            'email' => ['sometimes', 'string', 'email', 'max:255', Rule::unique('users')->ignore($staff->id)],
            'phone' => ['sometimes', 'string', 'max:20', Rule::unique('users')->ignore($staff->id)],
            'password' => 'sometimes|string|min:8|confirmed',
            'is_active' => 'sometimes|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $data = $request->only(['name', 'username', 'email', 'phone', 'is_active']);
        
        if ($request->filled('password')) {
            $data['password'] = Hash::make($request->password);
        }

        $staff->update($data);

        return response()->json([
            'success' => true,
            'message' => 'Staff member updated successfully',
            'data' => $staff,
        ]);
    }

    /**
     * Remove the specified staff member
     */
    public function destroy($id): JsonResponse
    {
        $staff = User::where('role', 'staff')->find($id);

        if (!$staff) {
            return response()->json([
                'success' => false,
                'message' => 'Staff member not found'
            ], 404);
        }

        $staff->delete();

        return response()->json([
            'success' => true,
            'message' => 'Staff member deleted successfully',
        ]);
    }

    /**
     * Toggle staff member active status
     */
    public function toggleStatus($id): JsonResponse
    {
        $staff = User::where('role', 'staff')->find($id);

        if (!$staff) {
            return response()->json([
                'success' => false,
                'message' => 'Staff member not found'
            ], 404);
        }

        $staff->is_active = !$staff->is_active;
        $staff->save();

        return response()->json([
            'success' => true,
            'message' => 'Staff status updated successfully',
            'data' => $staff,
        ]);
    }

    /**
     * Get all active staff members
     */
    public function activeStaff(): JsonResponse
    {
        $staff = User::where('role', 'staff')
                    ->where('is_active', true)
                    ->paginate(15);
        
        return response()->json([
            'success' => true,
            'data' => $staff,
        ]);
    }

    /**
     * Get all inactive staff members
     */
    public function inactiveStaff(): JsonResponse
    {
        $staff = User::where('role', 'staff')
                    ->where('is_active', false)
                    ->paginate(15);
        
        return response()->json([
            'success' => true,
            'data' => $staff,
        ]);
    }
}