<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\TransactionCategory;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class TransactionCategoryController extends Controller
{
    /**
     * Get all transaction categories
     */
    public function index(Request $request): JsonResponse
    {
        $query = TransactionCategory::query();

        // Filter by type (income/expense)
        if ($request->has('type')) {
            $query->where('type', $request->type);
        }

        // Filter by active status
        if ($request->has('is_active')) {
            $query->where('is_active', $request->boolean('is_active'));
        }

        // Get categories with parent-child relationship
        $categories = $query->with('parent')
            ->orderBy('name')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $categories,
        ]);
    }

    /**
     * Create a new transaction category
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255|unique:transaction_categories,name',
            'type' => 'required|in:income,expense',
            'parent_id' => 'nullable|exists:transaction_categories,id',
            'description' => 'nullable|string',
            'color' => 'nullable|string|size:7|regex:/^#[0-9A-Fa-f]{6}$/', // hex color
            'icon' => 'nullable|string|max:50',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $category = TransactionCategory::create([
                'name' => $request->name,
                'slug' => Str::slug($request->name),
                'type' => $request->type,
                'parent_id' => $request->parent_id,
                'description' => $request->description,
                'color' => $request->color,
                'icon' => $request->icon,
                'is_active' => $request->is_active ?? true,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Category created successfully',
                'data' => $category->load('parent'),
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create category: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get a single category
     */
    public function show($id): JsonResponse
    {
        $category = TransactionCategory::with(['parent', 'children'])->find($id);

        if (!$category) {
            return response()->json([
                'success' => false,
                'message' => 'Category not found',
            ], 404);
        }

        // Get transaction count for this category
        $transactionCount = $category->transactions()->count();
        $totalAmount = $category->transactions()->sum('amount');

        return response()->json([
            'success' => true,
            'data' => [
                'category' => $category,
                'stats' => [
                    'transaction_count' => $transactionCount,
                    'total_amount' => $totalAmount,
                ],
            ],
        ]);
    }

    /**
     * Update a category
     */
    public function update(Request $request, $id): JsonResponse
    {
        $category = TransactionCategory::find($id);

        if (!$category) {
            return response()->json([
                'success' => false,
                'message' => 'Category not found',
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255|unique:transaction_categories,name,' . $id,
            'type' => 'required|in:income,expense',
            'parent_id' => 'nullable|exists:transaction_categories,id|not_in:' . $id,
            'description' => 'nullable|string',
            'color' => 'nullable|string|size:7|regex:/^#[0-9A-Fa-f]{6}$/',
            'icon' => 'nullable|string|max:50',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $category->update([
                'name' => $request->name,
                'slug' => Str::slug($request->name),
                'type' => $request->type,
                'parent_id' => $request->parent_id,
                'description' => $request->description,
                'color' => $request->color,
                'icon' => $request->icon,
                'is_active' => $request->is_active ?? $category->is_active,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Category updated successfully',
                'data' => $category->load('parent'),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update category: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Delete a category
     */
    public function destroy($id): JsonResponse
    {
        $category = TransactionCategory::find($id);

        if (!$category) {
            return response()->json([
                'success' => false,
                'message' => 'Category not found',
            ], 404);
        }

        // Check if category has transactions
        $hasTransactions = $category->transactions()->exists();
        if ($hasTransactions) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete category that has transactions. Please reassign transactions first.',
            ], 400);
        }

        // Check if category has children
        $hasChildren = $category->children()->exists();
        if ($hasChildren) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete category that has subcategories. Please delete subcategories first.',
            ], 400);
        }

        try {
            $category->delete();

            return response()->json([
                'success' => true,
                'message' => 'Category deleted successfully',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete category: ' . $e->getMessage(),
            ], 500);
        }
    }
}
