<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Transaction;
use App\Models\Account;
use App\Services\TransactionService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class TransactionController extends Controller
{
    protected $transactionService;

    public function __construct(TransactionService $transactionService)
    {
        $this->transactionService = $transactionService;
    }

    /**
     * Get all transactions with filters
     */
    public function index(Request $request): JsonResponse
    {
        $query = Transaction::with(['account', 'category', 'creator']);

        // Filter by type
        if ($request->has('type')) {
            $query->where('type', $request->type);
        }

        // Filter by direction (credit/debit)
        if ($request->has('direction')) {
            $query->where('direction', $request->direction);
        }

        // Filter by account
        if ($request->has('account_id')) {
            $query->where('account_id', $request->account_id);
        }

        // Filter by payment method
        if ($request->has('payment_method')) {
            $query->where('payment_method', $request->payment_method);
        }

        // Filter by status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Filter by date range
        if ($request->has('start_date') && $request->has('end_date')) {
            $query->whereBetween('transaction_date', [$request->start_date, $request->end_date]);
        }

        // Filter by party type and id
        if ($request->has('party_type') && $request->has('party_id')) {
            $query->where('party_type', $request->party_type)
                  ->where('party_id', $request->party_id);
        }

        // Search
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('transaction_number', 'like', "%{$search}%")
                  ->orWhere('reference_number', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }

        $transactions = $query->orderBy('transaction_date', 'desc')
                              ->orderBy('created_at', 'desc')
                              ->paginate(50);

        return response()->json([
            'success' => true,
            'data' => $transactions,
        ]);
    }

    /**
     * Record an expense
     */
    public function recordExpense(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'amount' => 'required|numeric|min:0.01',
            'payment_method' => 'required|string',
            'transaction_date' => 'required|date',
            'category_id' => 'nullable|exists:transaction_categories,id',
            'account_id' => 'nullable|exists:accounts,id',
            'description' => 'nullable|string',
            'reference_number' => 'nullable|string',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $transaction = $this->transactionService->recordExpense(
                amount: $request->amount,
                paymentMethod: $request->payment_method,
                transactionDate: $request->transaction_date,
                categoryId: $request->category_id,
                accountId: $request->account_id,
                description: $request->description,
                referenceNumber: $request->reference_number,
                notes: $request->notes
            );

            return response()->json([
                'success' => true,
                'message' => 'Expense recorded successfully',
                'data' => $transaction->load(['account', 'category']),
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to record expense: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Record other income
     */
    public function recordIncome(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'amount' => 'required|numeric|min:0.01',
            'payment_method' => 'required|string',
            'transaction_date' => 'required|date',
            'category_id' => 'nullable|exists:transaction_categories,id',
            'account_id' => 'nullable|exists:accounts,id',
            'description' => 'nullable|string',
            'reference_number' => 'nullable|string',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $transaction = $this->transactionService->recordIncome(
                amount: $request->amount,
                paymentMethod: $request->payment_method,
                transactionDate: $request->transaction_date,
                categoryId: $request->category_id,
                accountId: $request->account_id,
                description: $request->description,
                referenceNumber: $request->reference_number,
                notes: $request->notes
            );

            return response()->json([
                'success' => true,
                'message' => 'Income recorded successfully',
                'data' => $transaction->load(['account', 'category']),
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to record income: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Transfer between accounts
     */
    public function transfer(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'from_account_id' => 'required|exists:accounts,id',
            'to_account_id' => 'required|exists:accounts,id|different:from_account_id',
            'amount' => 'required|numeric|min:0.01',
            'transaction_date' => 'required|date',
            'description' => 'nullable|string',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $transactions = $this->transactionService->recordTransfer(
                fromAccountId: $request->from_account_id,
                toAccountId: $request->to_account_id,
                amount: $request->amount,
                transactionDate: $request->transaction_date,
                description: $request->description,
                notes: $request->notes
            );

            return response()->json([
                'success' => true,
                'message' => 'Transfer completed successfully',
                'data' => $transactions,
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to complete transfer: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get transaction statistics
     */
    public function stats(Request $request): JsonResponse
    {
        $startDate = $request->get('start_date', now()->startOfMonth()->toDateString());
        $endDate = $request->get('end_date', now()->toDateString());
        $accountId = $request->get('account_id');

        $query = Transaction::completed()->dateRange($startDate, $endDate);

        if ($accountId) {
            $query->forAccount($accountId);
        }

        $credits = (clone $query)->credits()->sum('amount');
        $debits = (clone $query)->debits()->sum('amount');
        $netCashFlow = $credits - $debits;

        // Get totals by type
        $salePayments = (clone $query)->ofType('sale_payment')->sum('amount');
        $purchasePayments = (clone $query)->ofType('purchase_payment')->sum('amount');
        $installmentPayments = (clone $query)->ofType('installment_payment')->sum('amount');
        $expenses = (clone $query)->ofType('expense')->sum('amount');
        $otherIncome = (clone $query)->ofType('income')->sum('amount');

        // Account balances
        $accountsQuery = Account::active();
        if ($accountId) {
            $accountsQuery->where('id', $accountId);
        }
        $accounts = $accountsQuery->get()->map(function ($account) {
            return [
                'id' => $account->id,
                'name' => $account->name,
                'type' => $account->type,
                'balance' => $account->current_balance,
            ];
        });

        return response()->json([
            'success' => true,
            'data' => [
                'period' => [
                    'start_date' => $startDate,
                    'end_date' => $endDate,
                ],
                'summary' => [
                    'total_credits' => $credits,
                    'total_debits' => $debits,
                    'net_cash_flow' => $netCashFlow,
                ],
                'by_type' => [
                    'sale_payments' => $salePayments,
                    'purchase_payments' => $purchasePayments,
                    'installment_payments' => $installmentPayments,
                    'expenses' => $expenses,
                    'other_income' => $otherIncome,
                ],
                'accounts' => $accounts,
                'total_balance' => $accounts->sum('balance'),
            ],
        ]);
    }

    /**
     * Get cash flow report
     */
    public function cashFlow(Request $request): JsonResponse
    {
        $startDate = $request->get('start_date', now()->startOfMonth()->toDateString());
        $endDate = $request->get('end_date', now()->toDateString());
        $accountId = $request->get('account_id');
        $groupBy = $request->get('group_by', 'day'); // day, week, month

        $query = Transaction::completed()
            ->dateRange($startDate, $endDate)
            ->select(
                DB::raw('DATE(transaction_date) as date'),
                DB::raw('SUM(CASE WHEN direction = "credit" THEN amount ELSE 0 END) as credits'),
                DB::raw('SUM(CASE WHEN direction = "debit" THEN amount ELSE 0 END) as debits'),
                DB::raw('SUM(CASE WHEN direction = "credit" THEN amount ELSE -amount END) as net')
            );

        if ($accountId) {
            $query->forAccount($accountId);
        }

        $query->groupBy('date')
              ->orderBy('date', 'asc');

        $cashFlow = $query->get();

        return response()->json([
            'success' => true,
            'data' => $cashFlow,
        ]);
    }

    /**
     * Get a single transaction
     */
    public function show($id): JsonResponse
    {
        $transaction = Transaction::with(['account', 'category', 'creator', 'related'])
            ->find($id);

        if (!$transaction) {
            return response()->json([
                'success' => false,
                'message' => 'Transaction not found',
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $transaction,
        ]);
    }

    /**
     * Cancel a transaction
     */
    public function cancel($id): JsonResponse
    {
        $transaction = Transaction::find($id);

        if (!$transaction) {
            return response()->json([
                'success' => false,
                'message' => 'Transaction not found',
            ], 404);
        }

        if ($transaction->status === 'cancelled') {
            return response()->json([
                'success' => false,
                'message' => 'Transaction is already cancelled',
            ], 400);
        }

        DB::beginTransaction();
        try {
            $transaction->status = 'cancelled';
            $transaction->save();

            // Update account balance
            if ($transaction->account_id) {
                $account = Account::find($transaction->account_id);
                if ($account) {
                    $account->updateBalance();
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Transaction cancelled successfully',
                'data' => $transaction,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to cancel transaction: ' . $e->getMessage(),
            ], 500);
        }
    }
}
