<?php

namespace App\Mail;

use Illuminate\Bus\Queueable;
use Illuminate\Mail\Mailable;
use Illuminate\Mail\Mailables\Content;
use Illuminate\Mail\Mailables\Envelope;
use Illuminate\Mail\Mailables\Attachment;
use Illuminate\Queue\SerializesModels;
use Carbon\Carbon;

class ReportMail extends Mailable
{
    use Queueable, SerializesModels;

    public $reportType;
    public $filePath;
    public $startDate;
    public $endDate;

    /**
     * Create a new message instance.
     */
    public function __construct(
        string $reportType,
        string $filePath,
        Carbon $startDate,
        Carbon $endDate
    ) {
        $this->reportType = $reportType;
        $this->filePath = $filePath;
        $this->startDate = $startDate;
        $this->endDate = $endDate;
    }

    /**
     * Get the message envelope.
     */
    public function envelope(): Envelope
    {
        return new Envelope(
            subject: $this->getSubject(),
        );
    }

    /**
     * Get the message content definition.
     */
    public function content(): Content
    {
        return new Content(
            view: 'emails.report',
            with: [
                'reportType' => $this->reportType,
                'reportTitle' => $this->getReportTitle(),
                'period' => $this->getPeriodText(),
            ]
        );
    }

    /**
     * Get the attachments for the message.
     */
    public function attachments(): array
    {
        if (!file_exists($this->filePath)) {
            return [];
        }

        return [
            Attachment::fromPath($this->filePath)
                ->as(basename($this->filePath))
                ->withMime($this->getMimeType())
        ];
    }

    /**
     * Get email subject
     */
    private function getSubject(): string
    {
        $title = $this->getReportTitle();
        $period = $this->getPeriodText();
        
        return "{$title} - {$period}";
    }

    /**
     * Get report title
     */
    private function getReportTitle(): string
    {
        return match ($this->reportType) {
            'sales-summary' => 'Sales Summary Report',
            'sales-by-customer' => 'Sales by Customer Report',
            'sales-by-product' => 'Sales by Product Report',
            'payment-received' => 'Payment Received Report',
            'purchase-summary' => 'Purchase Summary Report',
            'purchase-by-supplier' => 'Purchase by Supplier Report',
            'payment-made' => 'Payment Made Report',
            'stock-summary' => 'Stock Summary Report',
            'low-stock-alert' => 'Low Stock Alert Report',
            'stock-movement' => 'Stock Movement Report',
            'profit-loss' => 'Profit & Loss Statement',
            'cash-flow' => 'Cash Flow Report',
            'tax-summary' => 'Tax Summary Report',
            'outstanding-receivables' => 'Outstanding Receivables Report',
            'outstanding-payables' => 'Outstanding Payables Report',
            'all-reports' => 'All Reports Package',
            default => 'Business Report'
        };
    }

    /**
     * Get period text
     */
    private function getPeriodText(): string
    {
        return $this->startDate->format('M d, Y') . ' - ' . $this->endDate->format('M d, Y');
    }

    /**
     * Get MIME type based on file extension
     */
    private function getMimeType(): string
    {
        $extension = pathinfo($this->filePath, PATHINFO_EXTENSION);
        
        return match ($extension) {
            'pdf' => 'application/pdf',
            'xlsx' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'csv' => 'text/csv',
            'zip' => 'application/zip',
            default => 'application/octet-stream'
        };
    }
}