<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Customer extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'email',
        'phone',
        'secondary_phone',
        'address',
        'city',
        'country',
        'cnic',
        'date_of_birth',
        'customer_type',
        'credit_limit',
        'notes', // Keep this for backward compatibility if needed
        'is_active',
    ];

    protected $casts = [
        'date_of_birth' => 'date',
        'credit_limit' => 'decimal:2',
        'is_active' => 'boolean',
    ];

    protected $attributes = [
        'is_active' => true,
        'customer_type' => 'retail',
        'credit_limit' => 0,
    ];

    /**
     * Get all notes for this customer
     */
    public function customerNotes()
    {
        return $this->hasMany(CustomerNote::class)->orderBy('created_at', 'desc');
    }

    /**
     * Get all sales for this customer
     */
    public function sales()
    {
        return $this->hasMany(Sale::class);
    }

    /**
     * Get total purchases amount
     */
    public function getTotalPurchasesAttribute()
    {
        return $this->sales()->sum('total_amount') ?? 0;
    }

    /**
     * Get total due amount
     */
    public function getTotalDueAttribute()
    {
        return $this->sales()
            ->whereIn('payment_status', ['pending', 'partial'])
            ->sum('due_amount') ?? 0;
    }

    /**
     * Get total paid amount
     */
    public function getTotalPaidAttribute()
    {
        return $this->sales()->sum('paid_amount') ?? 0;
    }

    /**
     * Get number of purchases
     */
    public function getPurchaseCountAttribute()
    {
        return $this->sales()->count();
    }

    /**
     * Check if customer has outstanding dues
     */
    public function hasDues()
    {
        return $this->total_due > 0;
    }

    /**
     * Check if customer has reached credit limit
     */
    public function hasReachedCreditLimit()
    {
        if (!$this->credit_limit || $this->credit_limit == 0) {
            return false;
        }

        return $this->total_due >= $this->credit_limit;
    }

    /**
     * Get available credit
     */
    public function getAvailableCreditAttribute()
    {
        if (!$this->credit_limit || $this->credit_limit == 0) {
            return 0;
        }

        return max(0, $this->credit_limit - $this->total_due);
    }

    /**
     * Scope to filter active customers
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope to filter inactive customers
     */
    public function scopeInactive($query)
    {
        return $query->where('is_active', false);
    }

    /**
     * Scope to filter by customer type
     */
    public function scopeOfType($query, $type)
    {
        return $query->where('customer_type', $type);
    }

    /**
     * Scope to filter customers with dues
     */
    public function scopeWithDues($query)
    {
        return $query->whereHas('sales', function($q) {
            $q->whereIn('payment_status', ['pending', 'partial']);
        });
    }

    /**
     * Scope to search customers
     */
    public function scopeSearch($query, $search)
    {
        return $query->where(function($q) use ($search) {
            $q->where('name', 'like', "%{$search}%")
              ->orWhere('phone', 'like', "%{$search}%")
              ->orWhere('email', 'like', "%{$search}%")
              ->orWhere('cnic', 'like', "%{$search}%");
        });
    }

    /**
     * Get customer display name with type
     */
    public function getDisplayNameAttribute()
    {
        $type = ucfirst($this->customer_type);
        return "{$this->name} ({$type})";
    }

    /**
     * Get customer type label
     */
    public function getCustomerTypeLabelAttribute()
    {
        return ucfirst($this->customer_type);
    }

    /**
     * Get status text
     */
    public function getStatusTextAttribute()
    {
        return $this->is_active ? 'Active' : 'Inactive';
    }
}