<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Inventory extends Model
{
    protected $table = 'inventory';

    protected $fillable = [
        'sku', 'brand_id', 'category_id', 'model', 
        'color', 'storage', 'ram',
        'purchase_price', 'selling_price', 'wholesale_price',
        'minimum_stock', 'description', 'condition',
        'warranty_period', 'camera', 'battery', 'screen_size',
        'processor', 'operating_system', 'supplier_id', 'status',
        'is_active', 'image'
    ];

    protected $casts = [
        'purchase_price' => 'decimal:2',
        'selling_price' => 'decimal:2',
        'wholesale_price' => 'decimal:2',
        'minimum_stock' => 'integer',
        'is_active' => 'boolean',
    ];

    // Add these to append to JSON responses
    protected $appends = [
        'quantity', // Total devices
        'available_items_count',
        'sold_items_count',
        'reserved_items_count',
        'total_value',
        'available_value',
    ];

    // Relationships
    public function brand(): BelongsTo
    {
        return $this->belongsTo(Brand::class);
    }

    public function category(): BelongsTo
    {
        return $this->belongsTo(Category::class);
    }

    public function supplier(): BelongsTo
    {
        return $this->belongsTo(Supplier::class);
    }

    public function items(): HasMany
    {
        return $this->hasMany(InventoryItem::class);
    }

    public function purchaseItems(): HasMany
    {
        return $this->hasMany(PurchaseItem::class);
    }

    public function saleItems(): HasMany
    {
        return $this->hasMany(SaleItem::class);
    }

    // Accessors for calculated fields
    public function getQuantityAttribute(): int
    {
        return $this->items()->count();
    }

    public function getAvailableItemsCountAttribute(): int
    {
        return $this->items()->where('status', 'available')->count();
    }

    public function getSoldItemsCountAttribute(): int
    {
        return $this->items()->where('status', 'sold')->count();
    }

    public function getReservedItemsCountAttribute(): int
    {
        return $this->items()->where('status', 'reserved')->count();
    }

    public function getTotalValueAttribute(): float
    {
        return $this->items()->sum('purchase_price');
    }

    public function getAvailableValueAttribute(): float
    {
        return $this->items()
            ->where('status', 'available')
            ->sum('purchase_price');
    }

    // Scopes
    public function scopeLowStock($query)
    {
        return $query->whereHas('items', function($q) {
            $q->where('status', 'available');
        }, '<=', function($query) {
            return $query->selectRaw('minimum_stock');
        })->where(function($q) {
            $q->whereHas('items', function($subQ) {
                $subQ->where('status', 'available');
            });
        });
    }

    public function scopeOutOfStock($query)
    {
        return $query->whereDoesntHave('items', function($q) {
            $q->where('status', 'available');
        });
    }

    public function scopeAvailable($query)
    {
        return $query->where('status', 'available')
                     ->whereHas('items', function($q) {
                         $q->where('status', 'available');
                     })
                     ->where('is_active', true);
    }

    // Helper method to check if low stock
    public function isLowStock(): bool
    {
        return $this->available_items_count <= $this->minimum_stock && $this->available_items_count > 0;
    }

    // Helper method to check if out of stock
    public function isOutOfStock(): bool
    {
        return $this->available_items_count === 0;
    }

    // Auto-update status based on available items
    public function updateStockStatus(): void
    {
        $availableCount = $this->available_items_count;
        
        if ($availableCount === 0) {
            $this->status = 'out_of_stock';
        } elseif ($this->status === 'out_of_stock') {
            $this->status = 'available';
        }
        
        $this->save();
    }
}