<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class InventoryItem extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'inventory_id',
        'purchase_item_id',
        'imei_1',
        'imei_2',
        'serial_number',
        'color',
        'storage',
        'ram',
        'purchase_price',
        'selling_price',
        'supplier_id',
        'purchase_date',
        'condition',
        'warranty_months',
        'warranty_expiry_date',
        'status',
        'location',
        'sold_date',
        'customer_id',
        'sale_id',
        'notes',
        'is_active'
    ];

    protected $casts = [
        'purchase_price' => 'decimal:2',
        'selling_price' => 'decimal:2',
        'purchase_date' => 'date',
        'warranty_expiry_date' => 'date',
        'sold_date' => 'datetime',
        'is_active' => 'boolean',
        'warranty_months' => 'integer',
    ];

    // Relationships
    public function inventory(): BelongsTo
    {
        return $this->belongsTo(Inventory::class);
    }

    public function supplier(): BelongsTo
    {
        return $this->belongsTo(Supplier::class);
    }

    public function customer(): BelongsTo
    {
        return $this->belongsTo(Customer::class);
    }

    // public function sale(): BelongsTo
    // {
    //     return $this->belongsTo(Sale::class);
    // }

    // Scopes
    public function scopeAvailable($query)
    {
        return $query->where('status', 'available')->where('is_active', true);
    }

    public function scopeSold($query)
    {
        return $query->where('status', 'sold');
    }

    public function scopeByImei($query, $imei)
    {
        return $query->where('imei_1', $imei);
    }

    // Accessors
    public function getProfitAttribute()
    {
        if ($this->status === 'sold') {
            return $this->selling_price - $this->purchase_price;
        }
        return null;
    }

    public function getWarrantyStatusAttribute()
    {
        if (!$this->warranty_expiry_date) {
            return 'no_warranty';
        }

        if ($this->warranty_expiry_date->isPast()) {
            return 'expired';
        }

        if ($this->warranty_expiry_date->diffInDays(now()) <= 30) {
            return 'expiring_soon';
        }

        return 'active';
    }
}
