<?php

namespace App\Services;

use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Purchase;
use App\Models\Inventory;
use App\Models\InventoryItem;
use App\Models\Customer;
use App\Models\Supplier;
use App\Models\ReportSchedule;
use App\Services\PdfExportService;
use App\Services\ExcelExportService;
use App\Services\CsvExportService;
use App\Mail\ReportMail;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;
use ZipArchive;

class ReportService
{
    protected $pdfExport;
    protected $excelExport;
    protected $csvExport;

    public function __construct(
        PdfExportService $pdfExport,
        ExcelExportService $excelExport,
        CsvExportService $csvExport
    ) {
        $this->pdfExport = $pdfExport;
        $this->excelExport = $excelExport;
        $this->csvExport = $csvExport;
    }

    /**
     * Get summary statistics
     */
    public function getSummary(Carbon $startDate, Carbon $endDate): array
    {
        // Total Revenue from orders
        $totalRevenue = Order::whereBetween('order_date', [$startDate, $endDate])
            ->where('status', '!=', 'cancelled')
            ->sum('grand_total');

        // Total Purchase Cost
        $totalPurchaseCost = Purchase::whereBetween('purchase_date', [$startDate, $endDate])
            ->sum('total_amount');

        // Net Profit
        $netProfit = $totalRevenue - $totalPurchaseCost;

        // Total Sales Count (completed orders)
        $totalSales = Order::whereBetween('order_date', [$startDate, $endDate])
            ->whereIn('status', ['confirmed', 'completed'])
            ->count();

        // Total Purchases Count
        $totalPurchases = Purchase::whereBetween('purchase_date', [$startDate, $endDate])
            ->count();

        return [
            'totalRevenue' => round($totalRevenue, 2),
            'netProfit' => round($netProfit, 2),
            'totalSales' => $totalSales,
            'totalPurchases' => $totalPurchases,
        ];
    }

    /**
     * Generate specific report
     */
    public function generateReport(string $type, Carbon $startDate, Carbon $endDate): array
    {
        return match ($type) {
            'sales-summary' => $this->getSalesSummary($startDate, $endDate),
            'sales-by-customer' => $this->getSalesByCustomer($startDate, $endDate),
            'sales-by-product' => $this->getSalesByProduct($startDate, $endDate),
            'payment-received' => $this->getPaymentReceived($startDate, $endDate),
            'purchase-summary' => $this->getPurchaseSummary($startDate, $endDate),
            'purchase-by-supplier' => $this->getPurchaseBySupplier($startDate, $endDate),
            'payment-made' => $this->getPaymentMade($startDate, $endDate),
            'stock-summary' => $this->getStockSummary(),
            'low-stock-alert' => $this->getLowStockAlert(),
            'stock-movement' => $this->getStockMovement($startDate, $endDate),
            'profit-loss' => $this->getProfitLoss($startDate, $endDate),
            'cash-flow' => $this->getCashFlow($startDate, $endDate),
            'tax-summary' => $this->getTaxSummary($startDate, $endDate),
            'outstanding-receivables' => $this->getOutstandingReceivables(),
            'outstanding-payables' => $this->getOutstandingPayables(),
            default => throw new \Exception('Invalid report type')
        };
    }

    /**
     * Export report to file
     */
    public function exportReport(string $type, string $format, Carbon $startDate, Carbon $endDate): string
    {
        $reportData = $this->generateReport($type, $startDate, $endDate);
        
        $fileName = $this->getFileName($type, $format);

        return match ($format) {
            'pdf' => $this->pdfExport->export($type, $reportData, $fileName),
            'excel' => $this->excelExport->export($type, $reportData, $fileName),
            'csv' => $this->csvExport->export($type, $reportData, $fileName),
            default => throw new \Exception('Invalid format')
        };
    }

    /**
     * Export all reports as ZIP
     */
    public function exportAllReports(string $format, Carbon $startDate, Carbon $endDate): string
    {
        $reportTypes = [
            'sales-summary',
            'purchase-summary',
            'stock-summary',
            'profit-loss',
            'cash-flow'
        ];

        $files = [];
        foreach ($reportTypes as $type) {
            try {
                $files[] = $this->exportReport($type, $format, $startDate, $endDate);
            } catch (\Exception $e) {
                \Log::error("Failed to export {$type}: " . $e->getMessage());
            }
        }

        return $this->createZipArchive($files);
    }

    /**
     * Email report
     */
    public function emailReport(string $email, string $type, Carbon $startDate, Carbon $endDate): void
    {
        $reportData = $this->generateReport($type, $startDate, $endDate);
        $pdfFile = $this->pdfExport->export($type, $reportData, $this->getFileName($type, 'pdf'));

        Mail::to($email)->send(new ReportMail($type, $pdfFile, $startDate, $endDate));

        // Clean up temporary file
        if (file_exists($pdfFile)) {
            unlink($pdfFile);
        }
    }

    /**
     * Email all reports
     */
    public function emailAllReports(string $email, Carbon $startDate, Carbon $endDate): void
    {
        $zipFile = $this->exportAllReports('pdf', $startDate, $endDate);

        Mail::to($email)->send(new ReportMail('all-reports', $zipFile, $startDate, $endDate));

        // Clean up temporary file
        if (file_exists($zipFile)) {
            unlink($zipFile);
        }
    }

    /**
     * Get scheduled reports
     */
    public function getScheduledReports(): array
    {
        return ReportSchedule::orderBy('created_at', 'desc')->get()->toArray();
    }

    /**
     * Create scheduled report
     */
    public function createSchedule(array $data): ReportSchedule
    {
        $schedule = ReportSchedule::create([
            'name' => $data['name'],
            'type' => $data['type'],
            'frequency' => $data['frequency'],
            'format' => $data['format'],
            'recipients' => $data['recipients'],
            'next_run' => $this->calculateNextRun($data['frequency'])
        ]);

        return $schedule;
    }

    /**
     * Update scheduled report
     */
    public function updateSchedule(int $id, array $data): ReportSchedule
    {
        $schedule = ReportSchedule::findOrFail($id);
        
        $schedule->update($data);

        if (isset($data['frequency'])) {
            $schedule->next_run = $this->calculateNextRun($data['frequency']);
            $schedule->save();
        }

        return $schedule;
    }

    /**
     * Delete scheduled report
     */
    public function deleteSchedule(int $id): void
    {
        ReportSchedule::findOrFail($id)->delete();
    }

    /**
     * Calculate next run date based on frequency
     */
    private function calculateNextRun(string $frequency): Carbon
    {
        return match ($frequency) {
            'daily' => Carbon::tomorrow(),
            'weekly' => Carbon::now()->addWeek()->startOfWeek(),
            'monthly' => Carbon::now()->addMonth()->startOfMonth(),
            default => Carbon::tomorrow()
        };
    }

    /**
     * Generate file name
     */
    private function getFileName(string $type, string $format): string
    {
        $extension = $format === 'excel' ? 'xlsx' : $format;
        $timestamp = now()->format('Y-m-d_His');
        return "{$type}_{$timestamp}.{$extension}";
    }

    /**
     * Create ZIP archive from files
     */
    private function createZipArchive(array $files): string
    {
        $zipFileName = storage_path('app/temp/all-reports-' . time() . '.zip');
        
        // Ensure temp directory exists
        if (!file_exists(dirname($zipFileName))) {
            mkdir(dirname($zipFileName), 0755, true);
        }

        $zip = new ZipArchive();
        if ($zip->open($zipFileName, ZipArchive::CREATE) === true) {
            foreach ($files as $file) {
                if (file_exists($file)) {
                    $zip->addFile($file, basename($file));
                }
            }
            $zip->close();
        }

        // Clean up individual files
        foreach ($files as $file) {
            if (file_exists($file)) {
                unlink($file);
            }
        }

        return $zipFileName;
    }

    // ===== REPORT GENERATION METHODS =====

    /**
     * Sales Summary Report (using Orders)
     */
    private function getSalesSummary(Carbon $startDate, Carbon $endDate): array
    {
        $orders = Order::with(['customer', 'items.inventoryItem'])
            ->whereBetween('order_date', [$startDate, $endDate])
            ->where('status', '!=', 'cancelled')
            ->orderBy('order_date', 'desc')
            ->get();

        $summary = [
            'total_sales' => $orders->count(),
            'total_amount' => $orders->sum('grand_total'),
            'total_paid' => $orders->sum('amount_paid'),
            'total_due' => $orders->sum(function($order) {
                return $order->grand_total - $order->amount_paid;
            }),
            'average_sale' => $orders->avg('grand_total'),
        ];

        return [
            'summary' => $summary,
            'sales' => $orders->map(function($order) {
                return [
                    'invoice_number' => $order->order_number,
                    'order_date' => $order->order_date,
                    'customer' => [
                        'name' => $order->customer ? $order->customer->name : 'Walk-in Customer',
                        'phone' => $order->customer ? $order->customer->phone : '-'
                    ],
                    'total_amount' => $order->grand_total,
                    'paid_amount' => $order->amount_paid,
                    'due_amount' => $order->grand_total - $order->amount_paid,
                    'status' => $order->status,
                    'payment_status' => $order->payment_status
                ];
            })->toArray(),
            'period' => [
                'start' => $startDate->format('Y-m-d'),
                'end' => $endDate->format('Y-m-d')
            ]
        ];
    }

    /**
     * Sales by Customer Report
     */
    private function getSalesByCustomer(Carbon $startDate, Carbon $endDate): array
    {
        $customerSales = Order::with('customer')
            ->whereBetween('order_date', [$startDate, $endDate])
            ->where('status', '!=', 'cancelled')
            ->get()
            ->groupBy('customer_id')
            ->map(function ($orders, $customerId) {
                $customer = $orders->first()->customer;
                return [
                    'customer_id' => $customerId,
                    'customer_name' => $customer ? $customer->name : 'Walk-in Customer',
                    'customer_phone' => $customer ? $customer->phone : '-',
                    'total_sales' => $orders->count(),
                    'total_amount' => $orders->sum('grand_total'),
                    'total_paid' => $orders->sum('amount_paid'),
                    'total_due' => $orders->sum(function($order) {
                        return $order->grand_total - $order->amount_paid;
                    }),
                ];
            })
            ->sortByDesc('total_amount')
            ->values()
            ->toArray();

        return [
            'customers' => $customerSales,
            'period' => [
                'start' => $startDate->format('Y-m-d'),
                'end' => $endDate->format('Y-m-d')
            ]
        ];
    }

    /**
     * Sales by Product Report
     */
    private function getSalesByProduct(Carbon $startDate, Carbon $endDate): array
    {
        $productSales = \DB::table('order_items')
            ->join('orders', 'order_items.order_id', '=', 'orders.id')
            ->join('inventory_items', 'order_items.inventory_item_id', '=', 'inventory_items.id')
            ->join('inventories', 'inventory_items.inventory_id', '=', 'inventories.id')
            ->whereBetween('orders.order_date', [$startDate, $endDate])
            ->where('orders.status', '!=', 'cancelled')
            ->select(
                'inventories.id as product_id',
                'inventories.brand',
                'inventories.model',
                \DB::raw('CONCAT(inventories.brand, " ", inventories.model) as product_name'),
                \DB::raw('SUM(order_items.qty) as total_quantity'),
                \DB::raw('SUM(order_items.total) as total_sales'),
                \DB::raw('AVG(order_items.unit_price) as average_price')
            )
            ->groupBy('inventories.id', 'inventories.brand', 'inventories.model')
            ->orderBy('total_sales', 'desc')
            ->get()
            ->toArray();

        return [
            'products' => $productSales,
            'period' => [
                'start' => $startDate->format('Y-m-d'),
                'end' => $endDate->format('Y-m-d')
            ]
        ];
    }

    /**
     * Payment Received Report
     */
    private function getPaymentReceived(Carbon $startDate, Carbon $endDate): array
    {
        $payments = \DB::table('order_payments')
            ->join('orders', 'order_payments.order_id', '=', 'orders.id')
            ->leftJoin('customers', 'orders.customer_id', '=', 'customers.id')
            ->whereBetween('order_payments.paid_at', [$startDate, $endDate])
            ->select(
                'order_payments.*',
                'orders.order_number',
                'customers.name as customer_name'
            )
            ->orderBy('order_payments.paid_at', 'desc')
            ->get()
            ->toArray();

        $summary = [
            'total_payments' => count($payments),
            'total_amount' => collect($payments)->sum('amount'),
            'payment_methods' => collect($payments)->groupBy('payment_method')
                ->map(function ($items) {
                    return [
                        'count' => $items->count(),
                        'amount' => $items->sum('amount')
                    ];
                })
        ];

        return [
            'summary' => $summary,
            'payments' => $payments,
            'period' => [
                'start' => $startDate->format('Y-m-d'),
                'end' => $endDate->format('Y-m-d')
            ]
        ];
    }

    /**
     * Purchase Summary Report
     */
    private function getPurchaseSummary(Carbon $startDate, Carbon $endDate): array
    {
        $purchases = Purchase::with(['supplier', 'items'])
            ->whereBetween('purchase_date', [$startDate, $endDate])
            ->orderBy('purchase_date', 'desc')
            ->get();

        $summary = [
            'total_purchases' => $purchases->count(),
            'total_amount' => $purchases->sum('total_amount'),
            'total_paid' => $purchases->sum('paid_amount'),
            'total_due' => $purchases->sum('due_amount'),
            'average_purchase' => $purchases->avg('total_amount'),
        ];

        return [
            'summary' => $summary,
            'purchases' => $purchases->toArray(),
            'period' => [
                'start' => $startDate->format('Y-m-d'),
                'end' => $endDate->format('Y-m-d')
            ]
        ];
    }

    /**
     * Purchase by Supplier Report
     */
    private function getPurchaseBySupplier(Carbon $startDate, Carbon $endDate): array
    {
        $supplierPurchases = Purchase::with('supplier')
            ->whereBetween('purchase_date', [$startDate, $endDate])
            ->get()
            ->groupBy('supplier_id')
            ->map(function ($purchases, $supplierId) {
                $supplier = $purchases->first()->supplier;
                return [
                    'supplier_id' => $supplierId,
                    'supplier_name' => $supplier ? $supplier->name : 'Unknown',
                    'supplier_phone' => $supplier ? $supplier->phone : '-',
                    'total_purchases' => $purchases->count(),
                    'total_amount' => $purchases->sum('total_amount'),
                    'total_paid' => $purchases->sum('paid_amount'),
                    'total_due' => $purchases->sum('due_amount'),
                ];
            })
            ->sortByDesc('total_amount')
            ->values()
            ->toArray();

        return [
            'suppliers' => $supplierPurchases,
            'period' => [
                'start' => $startDate->format('Y-m-d'),
                'end' => $endDate->format('Y-m-d')
            ]
        ];
    }

    /**
     * Payment Made Report
     */
    private function getPaymentMade(Carbon $startDate, Carbon $endDate): array
    {
        $payments = \DB::table('purchase_payments')
            ->join('purchases', 'purchase_payments.purchase_id', '=', 'purchases.id')
            ->leftJoin('suppliers', 'purchases.supplier_id', '=', 'suppliers.id')
            ->whereBetween('purchase_payments.payment_date', [$startDate, $endDate])
            ->select(
                'purchase_payments.*',
                'purchases.invoice_number',
                'suppliers.name as supplier_name'
            )
            ->orderBy('purchase_payments.payment_date', 'desc')
            ->get()
            ->toArray();

        $summary = [
            'total_payments' => count($payments),
            'total_amount' => collect($payments)->sum('amount'),
            'payment_methods' => collect($payments)->groupBy('payment_method')
                ->map(function ($items) {
                    return [
                        'count' => $items->count(),
                        'amount' => $items->sum('amount')
                    ];
                })
        ];

        return [
            'summary' => $summary,
            'payments' => $payments,
            'period' => [
                'start' => $startDate->format('Y-m-d'),
                'end' => $endDate->format('Y-m-d')
            ]
        ];
    }

    /**
     * Stock Summary Report
     */
    private function getStockSummary(): array
    {
        $inventories = Inventory::with(['items' => function ($query) {
            $query->where('status', 'available');
        }])->get();

        $stockData = $inventories->map(function ($inventory) {
            $totalStock = $inventory->items->count();
            $totalValue = $inventory->items->sum('purchase_price');

            return [
                'product_id' => $inventory->id,
                'product_name' => $inventory->brand . ' ' . $inventory->model,
                'brand' => $inventory->brand,
                'model' => $inventory->model,
                'total_stock' => $totalStock,
                'total_value' => $totalValue,
                'average_cost' => $totalStock > 0 ? $totalValue / $totalStock : 0,
            ];
        })->toArray();

        $summary = [
            'total_products' => count($stockData),
            'total_units' => collect($stockData)->sum('total_stock'),
            'total_value' => collect($stockData)->sum('total_value'),
        ];

        return [
            'summary' => $summary,
            'stock' => $stockData
        ];
    }

    /**
     * Low Stock Alert Report
     */
    private function getLowStockAlert(): array
    {
        $lowStockThreshold = 5;

        $inventories = Inventory::with(['items' => function ($query) {
            $query->where('status', 'available');
        }])->get();

        $lowStockProducts = $inventories->filter(function ($inventory) use ($lowStockThreshold) {
            return $inventory->items->count() <= $lowStockThreshold;
        })->map(function ($inventory) {
            $totalStock = $inventory->items->count();
            return [
                'product_id' => $inventory->id,
                'product_name' => $inventory->brand . ' ' . $inventory->model,
                'brand' => $inventory->brand,
                'model' => $inventory->model,
                'current_stock' => $totalStock,
                'status' => $totalStock === 0 ? 'Out of Stock' : 'Low Stock',
            ];
        })->values()->toArray();

        return [
            'low_stock_products' => $lowStockProducts,
            'threshold' => $lowStockThreshold
        ];
    }

    /**
     * Stock Movement Report
     */
    private function getStockMovement(Carbon $startDate, Carbon $endDate): array
    {
        // Items purchased/added
        $purchased = InventoryItem::with('inventory')
            ->whereBetween('created_at', [$startDate, $endDate])
            ->get()
            ->map(function($item) {
                return [
                    'product_name' => $item->inventory->brand . ' ' . $item->inventory->model,
                    'quantity' => 1,
                    'total_cost' => $item->purchase_price,
                    'type' => 'IN',
                    'date' => $item->created_at->format('Y-m-d')
                ];
            });

        // Items sold
        $sold = \DB::table('order_items')
            ->join('orders', 'order_items.order_id', '=', 'orders.id')
            ->join('inventory_items', 'order_items.inventory_item_id', '=', 'inventory_items.id')
            ->join('inventories', 'inventory_items.inventory_id', '=', 'inventories.id')
            ->whereBetween('orders.order_date', [$startDate, $endDate])
            ->where('orders.status', '!=', 'cancelled')
            ->select(
                \DB::raw('CONCAT(inventories.brand, " ", inventories.model) as product_name'),
                \DB::raw('SUM(order_items.qty) as quantity'),
                \DB::raw('SUM(order_items.total) as total_cost'),
                \DB::raw("'OUT' as type"),
                'orders.order_date as date'
            )
            ->groupBy('inventories.id', 'inventories.brand', 'inventories.model', 'orders.order_date')
            ->get();

        $movements = $purchased->merge($sold)->sortByDesc('date')->values()->toArray();

        return [
            'movements' => $movements,
            'period' => [
                'start' => $startDate->format('Y-m-d'),
                'end' => $endDate->format('Y-m-d')
            ]
        ];
    }

    /**
     * Profit & Loss Report
     */
    private function getProfitLoss(Carbon $startDate, Carbon $endDate): array
    {
        // Revenue from orders
        $totalRevenue = Order::whereBetween('order_date', [$startDate, $endDate])
            ->where('status', '!=', 'cancelled')
            ->sum('grand_total');

        // Cost of Goods Sold (COGS) - purchase price of sold items
        $cogs = \DB::table('order_items')
            ->join('orders', 'order_items.order_id', '=', 'orders.id')
            ->join('inventory_items', 'order_items.inventory_item_id', '=', 'inventory_items.id')
            ->whereBetween('orders.order_date', [$startDate, $endDate])
            ->where('orders.status', '!=', 'cancelled')
            ->sum('inventory_items.purchase_price');

        // Gross Profit
        $grossProfit = $totalRevenue - $cogs;

        // Operating Expenses (if you have an expenses table)
        $operatingExpenses = 0; // TODO: Add expenses calculation if needed

        // Net Profit
        $netProfit = $grossProfit - $operatingExpenses;

        return [
            'revenue' => round($totalRevenue, 2),
            'cogs' => round($cogs, 2),
            'gross_profit' => round($grossProfit, 2),
            'gross_profit_margin' => $totalRevenue > 0 ? round(($grossProfit / $totalRevenue) * 100, 2) : 0,
            'operating_expenses' => round($operatingExpenses, 2),
            'net_profit' => round($netProfit, 2),
            'net_profit_margin' => $totalRevenue > 0 ? round(($netProfit / $totalRevenue) * 100, 2) : 0,
            'period' => [
                'start' => $startDate->format('Y-m-d'),
                'end' => $endDate->format('Y-m-d')
            ]
        ];
    }

    /**
     * Cash Flow Report
     */
    private function getCashFlow(Carbon $startDate, Carbon $endDate): array
    {
        // Cash In (Order Payments)
        $cashIn = \DB::table('order_payments')
            ->whereBetween('paid_at', [$startDate, $endDate])
            ->sum('amount');

        // Cash Out (Purchase Payments)
        $cashOut = \DB::table('purchase_payments')
            ->whereBetween('payment_date', [$startDate, $endDate])
            ->sum('amount');

        // Net Cash Flow
        $netCashFlow = $cashIn - $cashOut;

        // Daily breakdown
        $dailyCashFlow = \DB::table('order_payments')
            ->whereBetween('paid_at', [$startDate, $endDate])
            ->select(
                \DB::raw('DATE(paid_at) as date'),
                \DB::raw('SUM(amount) as cash_in')
            )
            ->groupBy('date')
            ->get();

        return [
            'cash_in' => round($cashIn, 2),
            'cash_out' => round($cashOut, 2),
            'net_cash_flow' => round($netCashFlow, 2),
            'daily_breakdown' => $dailyCashFlow->toArray(),
            'period' => [
                'start' => $startDate->format('Y-m-d'),
                'end' => $endDate->format('Y-m-d')
            ]
        ];
    }

    /**
     * Tax Summary Report
     */
    private function getTaxSummary(Carbon $startDate, Carbon $endDate): array
    {
        // Tax from orders
        $salesTax = Order::whereBetween('order_date', [$startDate, $endDate])
            ->where('status', '!=', 'cancelled')
            ->sum('tax_total');

        // Tax from purchases (if applicable)
        $purchaseTax = Purchase::whereBetween('purchase_date', [$startDate, $endDate])
            ->sum('tax_amount');

        $netTax = $salesTax - $purchaseTax;

        return [
            'sales_tax_collected' => round($salesTax, 2),
            'purchase_tax_paid' => round($purchaseTax, 2),
            'net_tax_payable' => round($netTax, 2),
            'period' => [
                'start' => $startDate->format('Y-m-d'),
                'end' => $endDate->format('Y-m-d')
            ]
        ];
    }

    /**
     * Outstanding Receivables Report
     */
    private function getOutstandingReceivables(): array
    {
        $receivables = Order::with('customer')
            ->where('status', '!=', 'cancelled')
            ->whereRaw('grand_total > amount_paid')
            ->orderBy('order_date', 'desc')
            ->get()
            ->map(function ($order) {
                $dueAmount = $order->grand_total - $order->amount_paid;
                $daysOverdue = $order->due_date ? now()->diffInDays(Carbon::parse($order->due_date), false) : 0;
                
                return [
                    'invoice_number' => $order->order_number,
                    'customer_name' => $order->customer ? $order->customer->name : 'Walk-in',
                    'customer_phone' => $order->customer ? $order->customer->phone : '-',
                    'order_date' => $order->order_date,
                    'due_date' => $order->due_date,
                    'total_amount' => $order->grand_total,
                    'paid_amount' => $order->amount_paid,
                    'due_amount' => $dueAmount,
                    'days_overdue' => $daysOverdue > 0 ? $daysOverdue : 0,
                    'status' => $daysOverdue > 30 ? 'Critical' : ($daysOverdue > 0 ? 'Overdue' : 'Pending'),
                ];
            })
            ->toArray();

        $summary = [
            'total_receivables' => collect($receivables)->sum('due_amount'),
            'count' => count($receivables),
            'critical_count' => collect($receivables)->where('status', 'Critical')->count(),
            'overdue_count' => collect($receivables)->where('status', 'Overdue')->count(),
        ];

        return [
            'summary' => $summary,
            'receivables' => $receivables
        ];
    }

    /**
     * Outstanding Payables Report
     */
    private function getOutstandingPayables(): array
    {
        $payables = Purchase::with('supplier')
            ->where('due_amount', '>', 0)
            ->orderBy('purchase_date', 'desc')
            ->get()
            ->map(function ($purchase) {
                $daysOverdue = $purchase->due_date ? now()->diffInDays(Carbon::parse($purchase->due_date), false) : 0;
                
                return [
                    'invoice_number' => $purchase->invoice_number,
                    'supplier_name' => $purchase->supplier ? $purchase->supplier->name : 'Unknown',
                    'supplier_phone' => $purchase->supplier ? $purchase->supplier->phone : '-',
                    'purchase_date' => $purchase->purchase_date,
                    'due_date' => $purchase->due_date,
                    'total_amount' => $purchase->total_amount,
                    'paid_amount' => $purchase->paid_amount,
                    'due_amount' => $purchase->due_amount,
                    'days_overdue' => $daysOverdue > 0 ? $daysOverdue : 0,
                    'status' => $daysOverdue > 30 ? 'Critical' : ($daysOverdue > 0 ? 'Overdue' : 'Pending'),
                ];
            })
            ->toArray();

        $summary = [
            'total_payables' => collect($payables)->sum('due_amount'),
            'count' => count($payables),
            'critical_count' => collect($payables)->where('status', 'Critical')->count(),
            'overdue_count' => collect($payables)->where('status', 'Overdue')->count(),
        ];

        return [
            'summary' => $summary,
            'payables' => $payables
        ];
    }
}