<?php

namespace App\Services;

use App\Models\Transaction;
use App\Models\Account;
use Illuminate\Support\Facades\DB;
use Exception;

class TransactionService
{
    /**
     * Record a sale payment (customer pays you)
     */
    public function recordSalePayment(
        $orderId,
        float $amount,
        string $paymentMethod,
        string $transactionDate,
        ?int $customerId = null,
        ?int $accountId = null,
        ?string $referenceNumber = null,
        ?string $notes = null
    ): Transaction {
        return $this->createTransaction([
            'type' => 'sale_payment',
            'direction' => 'credit', // Money IN
            'amount' => $amount,
            'related_type' => 'App\Models\Order',
            'related_id' => $orderId,
            'payment_method' => $paymentMethod,
            'account_id' => $accountId ?? Account::getDefault()?->id,
            'party_type' => 'customer',
            'party_id' => $customerId,
            'transaction_date' => $transactionDate,
            'reference_number' => $referenceNumber,
            'status' => 'completed',
            'description' => "Payment received for order",
            'notes' => $notes,
        ]);
    }

    /**
     * Record a purchase payment (you pay supplier)
     */
    public function recordPurchasePayment(
        $purchaseId,
        float $amount,
        string $paymentMethod,
        string $transactionDate,
        ?int $supplierId = null,
        ?int $accountId = null,
        ?string $referenceNumber = null,
        ?string $notes = null
    ): Transaction {
        return $this->createTransaction([
            'type' => 'purchase_payment',
            'direction' => 'debit', // Money OUT
            'amount' => $amount,
            'related_type' => 'App\Models\Purchase',
            'related_id' => $purchaseId,
            'payment_method' => $paymentMethod,
            'account_id' => $accountId ?? Account::getDefault()?->id,
            'party_type' => 'supplier',
            'party_id' => $supplierId,
            'transaction_date' => $transactionDate,
            'reference_number' => $referenceNumber,
            'status' => 'completed',
            'description' => "Payment made to supplier",
            'notes' => $notes,
        ]);
    }

    /**
     * Record an installment payment
     */
    public function recordInstallmentPayment(
        $installmentId,
        float $amount,
        string $paymentMethod,
        string $transactionDate,
        ?int $customerId = null,
        ?int $accountId = null,
        ?string $referenceNumber = null,
        ?string $notes = null
    ): Transaction {
        return $this->createTransaction([
            'type' => 'installment_payment',
            'direction' => 'credit', // Money IN
            'amount' => $amount,
            'related_type' => 'App\Models\Installment',
            'related_id' => $installmentId,
            'payment_method' => $paymentMethod,
            'account_id' => $accountId ?? Account::getDefault()?->id,
            'party_type' => 'customer',
            'party_id' => $customerId,
            'transaction_date' => $transactionDate,
            'reference_number' => $referenceNumber,
            'status' => 'completed',
            'description' => "Installment payment received",
            'notes' => $notes,
        ]);
    }

    /**
     * Record an expense
     */
    public function recordExpense(
        float $amount,
        string $paymentMethod,
        string $transactionDate,
        ?int $categoryId = null,
        ?int $accountId = null,
        ?string $description = null,
        ?string $referenceNumber = null,
        ?string $notes = null,
        ?array $attachments = null
    ): Transaction {
        return $this->createTransaction([
            'type' => 'expense',
            'direction' => 'debit', // Money OUT
            'amount' => $amount,
            'payment_method' => $paymentMethod,
            'account_id' => $accountId ?? Account::getDefault()?->id,
            'category_id' => $categoryId,
            'transaction_date' => $transactionDate,
            'reference_number' => $referenceNumber,
            'status' => 'completed',
            'description' => $description ?? 'Business expense',
            'notes' => $notes,
            'attachments' => $attachments,
        ]);
    }

    /**
     * Record other income
     */
    public function recordIncome(
        float $amount,
        string $paymentMethod,
        string $transactionDate,
        ?int $categoryId = null,
        ?int $accountId = null,
        ?string $description = null,
        ?string $referenceNumber = null,
        ?string $notes = null
    ): Transaction {
        return $this->createTransaction([
            'type' => 'income',
            'direction' => 'credit', // Money IN
            'amount' => $amount,
            'payment_method' => $paymentMethod,
            'account_id' => $accountId ?? Account::getDefault()?->id,
            'category_id' => $categoryId,
            'transaction_date' => $transactionDate,
            'reference_number' => $referenceNumber,
            'status' => 'completed',
            'description' => $description ?? 'Other income',
            'notes' => $notes,
        ]);
    }

    /**
     * Transfer between accounts
     */
    public function recordTransfer(
        int $fromAccountId,
        int $toAccountId,
        float $amount,
        string $transactionDate,
        ?string $description = null,
        ?string $notes = null
    ): array {
        DB::beginTransaction();
        try {
            // Debit from source account
            $debit = $this->createTransaction([
                'type' => 'transfer',
                'direction' => 'debit',
                'amount' => $amount,
                'payment_method' => 'transfer',
                'account_id' => $fromAccountId,
                'transaction_date' => $transactionDate,
                'description' => $description ?? "Transfer to account",
                'notes' => $notes,
            ]);

            // Credit to destination account
            $credit = $this->createTransaction([
                'type' => 'transfer',
                'direction' => 'credit',
                'amount' => $amount,
                'payment_method' => 'transfer',
                'account_id' => $toAccountId,
                'transaction_date' => $transactionDate,
                'description' => $description ?? "Transfer from account",
                'notes' => $notes,
            ]);

            DB::commit();
            return ['debit' => $debit, 'credit' => $credit];
        } catch (Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Record a refund to customer
     */
    public function recordRefund(
        $relatedId,
        string $relatedType,
        float $amount,
        string $paymentMethod,
        string $transactionDate,
        ?int $customerId = null,
        ?int $accountId = null,
        ?string $referenceNumber = null,
        ?string $notes = null
    ): Transaction {
        return $this->createTransaction([
            'type' => 'refund',
            'direction' => 'debit', // Money OUT
            'amount' => $amount,
            'related_type' => $relatedType,
            'related_id' => $relatedId,
            'payment_method' => $paymentMethod,
            'account_id' => $accountId ?? Account::getDefault()?->id,
            'party_type' => 'customer',
            'party_id' => $customerId,
            'transaction_date' => $transactionDate,
            'reference_number' => $referenceNumber,
            'status' => 'completed',
            'description' => "Refund issued",
            'notes' => $notes,
        ]);
    }

    /**
     * Core method to create a transaction and update account balance
     */
    protected function createTransaction(array $data): Transaction
    {
        DB::beginTransaction();
        try {
            // Add created_by if not provided
            if (!isset($data['created_by'])) {
                $data['created_by'] = auth()->id();
            }

            // Create the transaction
            $transaction = Transaction::create($data);

            // Update account balance if account is specified
            if (isset($data['account_id']) && $data['account_id']) {
                $account = Account::find($data['account_id']);
                if ($account) {
                    $account->updateBalance();
                }
            }

            DB::commit();
            return $transaction;
        } catch (Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Get transactions for a specific related entity (Order, Purchase, etc.)
     */
    public function getTransactionsFor(string $relatedType, int $relatedId)
    {
        return Transaction::where('related_type', $relatedType)
            ->where('related_id', $relatedId)
            ->orderBy('transaction_date', 'desc')
            ->get();
    }

    /**
     * Get total paid for an entity
     */
    public function getTotalPaidFor(string $relatedType, int $relatedId): float
    {
        return Transaction::where('related_type', $relatedType)
            ->where('related_id', $relatedId)
            ->where('status', 'completed')
            ->sum('amount');
    }
}
