<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {

        // ============================================
        // STEP 1: CUSTOMERS TABLE
        // ============================================
        Schema::create('customers', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->string('email')->unique()->nullable();
            $table->string('phone')->unique();
            $table->string('secondary_phone')->nullable();
            $table->text('address')->nullable();
            $table->string('city')->nullable();
            $table->string('country')->nullable();
            $table->string('cnic')->nullable(); // National ID
            $table->date('date_of_birth')->nullable();
            $table->enum('customer_type', ['retail', 'wholesale', 'corporate'])->default('retail');
            $table->decimal('credit_limit', 10, 2)->default(0);
            $table->text('notes')->nullable();
            $table->boolean('is_active')->default(true);
            $table->timestamps();

            $table->index('phone');
            $table->index('email');
            $table->index('customer_type');
        });


        Schema::create('customer_notes', function (Blueprint $table) {
            $table->id();
            $table->foreignId('customer_id')->constrained()->onDelete('cascade');
            $table->text('note');
            $table->foreignId('created_by')->nullable()->constrained('users')->onDelete('set null');
            $table->timestamps();
            
            $table->index('customer_id');
        });

        

        // Create brands table
        Schema::create('brands', function (Blueprint $table) {
            $table->id();
            $table->string('name')->unique(); // Samsung, Apple, Xiaomi, etc.
            $table->string('slug')->unique(); // samsung, apple, xiaomi
            $table->text('description')->nullable();
            $table->string('logo')->nullable(); // Brand logo image path
            $table->string('country')->nullable(); // Country of origin
            $table->boolean('is_active')->default(true);
            $table->timestamps();
        });

        // Create categories table
        Schema::create('categories', function (Blueprint $table) {
            $table->id();
            $table->string('name')->unique(); // Smartphones, Feature Phones, Tablets, Accessories, etc.
            $table->string('slug')->unique(); // smartphones, feature-phones, tablets
            $table->text('description')->nullable();
            $table->boolean('is_active')->default(true);
            $table->timestamps();
        });

        // Create inventory table
        // Schema::create('inventory', function (Blueprint $table) {
        //     $table->id();

        //     // Product Information
        //     $table->string('sku')->unique(); // Stock Keeping Unit
        //     $table->foreignId('brand_id')->constrained('brands')->onDelete('cascade');
        //     $table->foreignId('category_id')->constrained('categories')->onDelete('cascade');
        //     $table->string('model'); // Galaxy S24, iPhone 15, etc.
        //     $table->string('color')->nullable();
        //     $table->string('storage')->nullable(); // 64GB, 128GB, 256GB, etc.

        //     // Pricing
        //     $table->decimal('purchase_price', 10, 2); // Cost price from supplier
        //     $table->decimal('selling_price', 10, 2); // Retail price
        //     $table->decimal('wholesale_price', 10, 2)->nullable(); // Optional wholesale price

        //     // Stock Management
        //     $table->integer('quantity')->default(0); // Available stock
        //     $table->integer('minimum_stock')->default(5); // Alert threshold

        //     // Product Details
        //     $table->text('description')->nullable();
        //     $table->enum('condition', ['new', 'refurbished', 'used'])->default('new');
        //     $table->string('warranty_period')->nullable(); // 1 year, 2 years, etc.

        //     // Specifications (Optional but useful for mobiles)
        //     $table->string('ram')->nullable(); // 4GB, 8GB, 12GB
        //     $table->string('camera')->nullable(); // 48MP, 108MP
        //     $table->string('battery')->nullable(); // 5000mAh
        //     $table->string('screen_size')->nullable(); // 6.5 inch
        //     $table->string('processor')->nullable(); // Snapdragon 8 Gen 2
        //     $table->string('operating_system')->nullable(); // Android 14, iOS 17

        //     // Supplier Relationship
        //     $table->foreignId('supplier_id')->nullable()->constrained('suppliers')->onDelete('set null');

        //     // Status
        //     $table->enum('status', ['available', 'out_of_stock', 'discontinued'])->default('available');
        //     $table->boolean('is_active')->default(true);

        //     // Images
        //     $table->string('image')->nullable(); // Main product image path

        //     $table->timestamps();

        //     // Indexes for faster queries
        //     $table->index('brand_id');
        //     $table->index('category_id');
        //     $table->index('model');
        //     $table->index('status');
        //     $table->index('sku');
        // });


        Schema::create('inventory', function (Blueprint $table) {
            $table->id();

            // Product Information
            $table->string('sku')->unique(); // Stock Keeping Unit
            $table->foreignId('brand_id')->constrained('brands')->onDelete('cascade');
            $table->foreignId('category_id')->constrained('categories')->onDelete('cascade');
            $table->string('model'); // Galaxy S24, iPhone 15, etc.

            // REMOVED: color, storage - these are now device-specific
            // Keep them as optional defaults that devices can inherit
            $table->string('color')->nullable(); // Default color for new devices
            $table->string('storage')->nullable(); // Default storage for new devices
            $table->string('ram')->nullable(); // Default RAM for new devices

            // Base Pricing (defaults for devices)
            $table->decimal('purchase_price', 10, 2); // Base cost price
            $table->decimal('selling_price', 10, 2); // Base retail price
            $table->decimal('wholesale_price', 10, 2)->nullable(); // Optional wholesale price

            // Stock Management
            // REMOVED: quantity - will be auto-calculated from inventory_items count
            $table->integer('minimum_stock')->default(5); // Alert threshold

            // Product Details
            $table->text('description')->nullable();
            $table->enum('condition', ['new', 'refurbished', 'used'])->default('new'); // Default condition
            $table->string('warranty_period')->nullable(); // Default warranty: 1 year, 2 years, etc.

            // Specifications (Product level - common to all devices)
            $table->string('camera')->nullable(); // 48MP, 108MP
            $table->string('battery')->nullable(); // 5000mAh
            $table->string('screen_size')->nullable(); // 6.5 inch
            $table->string('processor')->nullable(); // Snapdragon 8 Gen 2
            $table->string('operating_system')->nullable(); // Android 14, iOS 17

            // Supplier Relationship (default)
            $table->foreignId('supplier_id')->nullable()->constrained('suppliers')->onDelete('set null');

            // Status
            $table->enum('status', ['available', 'out_of_stock', 'discontinued'])->default('available');
            $table->boolean('is_active')->default(true);

            // Images
            $table->string('image')->nullable(); // Main product image path

            $table->timestamps();

            // Indexes for faster queries
            $table->index('brand_id');
            $table->index('category_id');
            $table->index('model');
            $table->index('status');
            $table->index('sku');
        });

        // Schema::create('inventory_items', function (Blueprint $table) {
        //     $table->id();

        //     // Link to parent product/inventory
        //     $table->foreignId('inventory_id')->constrained('inventory')->onDelete('cascade');

        //     // Unique Identifiers
        //     // $table->string('imei')->unique(); // Must be unique across all devices
        //     $table->string('imei_1')->unique(); // Must be unique across all devices
        //     $table->string('imei_2')->unique(); // Must be unique across all devices
        //     $table->string('serial_number')->nullable();

        //     // Device Specific Details (can override parent)
        //     $table->string('color')->nullable();
        //     $table->string('storage')->nullable(); // 128GB, 256GB
        //     $table->string('ram')->nullable(); // 6GB, 8GB

        //     // Pricing (can be different per device)
        //     $table->decimal('purchase_price', 10, 2);
        //     $table->decimal('selling_price', 10, 2);

        //     // Purchase Information
        //     $table->foreignId('supplier_id')->nullable()->constrained('suppliers')->onDelete('set null');
        //     $table->date('purchase_date')->nullable();

        //     // Condition & Warranty
        //     $table->enum('condition', ['new', 'refurbished', 'used'])->default('new');
        //     $table->integer('warranty_months')->nullable(); // 12, 24, etc.
        //     $table->date('warranty_expiry_date')->nullable();

        //     // Status Tracking
        //     $table->enum('status', ['available', 'sold', 'reserved', 'damaged', 'in_repair', 'returned'])->default('available');

        //     // Location
        //     $table->string('location')->nullable(); // Shelf A3, Warehouse B, etc.

        //     // Sale Information
        //     $table->dateTime('sold_date')->nullable();
        //     $table->foreignId('customer_id')->nullable()->constrained('customers')->onDelete('set null');
        //     // $table->foreignId('sale_id')->nullable()->constrained('sales')->onDelete('set null');

        //     // Additional Info
        //     $table->text('notes')->nullable();
        //     $table->boolean('is_active')->default(true);

        //     $table->timestamps();
        //     $table->softDeletes(); // Keep history of deleted items

        //     // Indexes
        //     $table->index('inventory_id');
        //     $table->index('imei_1');
        //     $table->index('imei_2');
        //     $table->index('status');
        //     $table->index('supplier_id');
        //     $table->index('sold_date');
        // });


        Schema::create('inventory_items', function (Blueprint $table) {
            $table->id();

            // Parent product
            $table->foreignId('inventory_id')
                ->constrained('inventory')
                ->onDelete('cascade');

            // Link to purchase item
            $table->foreignId('purchase_item_id')
                ->nullable();
                // ->constrained('purchase_items')
                // ->nullOnDelete();

            // Identifiers
            $table->string('imei_1')->unique();
            $table->string('imei_2')->nullable()->unique();
            $table->string('serial_number')->nullable();

            // Device attributes
            $table->string('color')->nullable();
            $table->string('storage')->nullable();
            $table->string('ram')->nullable();

            // Pricing
            $table->decimal('purchase_price', 10, 2);
            $table->decimal('selling_price', 10, 2);

            // Purchase info
            $table->foreignId('supplier_id')->nullable()->constrained('suppliers')->nullOnDelete();
            $table->date('purchase_date')->nullable();

            // Condition & Warranty
            $table->enum('condition', ['new', 'refurbished', 'used'])->default('new');
            $table->integer('warranty_months')->nullable();
            $table->date('warranty_expiry_date')->nullable();

            // Status
            $table->enum('status', ['available', 'sold', 'reserved', 'damaged', 'in_repair', 'returned'])
                ->default('available');

            // Location
            $table->string('location')->nullable();

            // Sale
            $table->dateTime('sold_date')->nullable();
            $table->foreignId('customer_id')->nullable()->constrained('customers')->nullOnDelete();

            // Meta
            $table->text('notes')->nullable();
            $table->boolean('is_active')->default(true);

            $table->timestamps();
            $table->softDeletes();

            // Indexes
            $table->index('inventory_id');
            $table->index('imei_1');
            $table->index('imei_2');
            $table->index('status');
            $table->index('supplier_id');
            $table->index('sold_date');
        });




        Schema::create('settings', function (Blueprint $table) {
            $table->id();
            $table->string('key')->unique(); // invoice_prefix, tax_rate, etc.
            $table->text('value')->nullable();
            $table->string('type')->default('string'); // string, number, boolean, json
            $table->string('group')->default('general'); // general, invoice, payment, etc.
            $table->text('description')->nullable();
            $table->timestamps();

            // Index for faster queries
            $table->index('key');
            $table->index('group');
        });

        // Insert default settings
        DB::table('settings')->insert([
            [
                'key' => 'invoice_prefix',
                'value' => 'INV',
                'type' => 'string',
                'group' => 'invoice',
                'description' => 'Prefix for invoice numbers',
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'key' => 'purchase_prefix',
                'value' => 'PUR',
                'type' => 'string',
                'group' => 'purchase',
                'description' => 'Prefix for purchase numbers',
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'key' => 'tax_rate',
                'value' => '0',
                'type' => 'number',
                'group' => 'general',
                'description' => 'Tax rate percentage',
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'key' => 'currency',
                'value' => 'PKR',
                'type' => 'string',
                'group' => 'general',
                'description' => 'Default currency',
                'created_at' => now(),
                'updated_at' => now(),
            ],
        ]);
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('settings');
    }
};
