<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;


return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('orders', function (Blueprint $table) {
            $table->id();
            $table->string('order_number')->unique();

            $table->foreignId('customer_id')
                ->nullable()
                ->constrained('customers')
                ->onDelete('set null');

            $table->enum('status', ['draft', 'confirmed', 'completed', 'cancelled'])
                ->default('draft');

            $table->enum('payment_status', ['unpaid', 'partial', 'paid', 'refunded'])
                ->default('unpaid');

            $table->date('order_date');
            $table->date('due_date')->nullable();

            $table->decimal('sub_total', 10, 2)->default(0);
            $table->decimal('discount_total', 10, 2)->default(0);
            $table->decimal('tax_total', 10, 2)->default(0);
            $table->decimal('grand_total', 10, 2)->default(0);
            $table->decimal('amount_paid', 10, 2)->default(0);

            $table->text('notes')->nullable();

            $table->timestamps();

            $table->index('customer_id');
            $table->index('status');
            $table->index('payment_status');
        });

        Schema::create('order_items', function (Blueprint $table) {
            $table->id();

            $table->foreignId('order_id')
                ->constrained('orders')
                ->onDelete('cascade');

            $table->foreignId('inventory_id')
                ->constrained('inventory')
                ->onDelete('restrict');

            $table->foreignId('inventory_item_id')
                ->constrained('inventory_items')
                ->onDelete('restrict');

            $table->string('description')->nullable();

            $table->integer('qty')->default(1);
            $table->decimal('unit_price', 10, 2);
            $table->decimal('discount_amount', 10, 2)->default(0);
            $table->decimal('tax_amount', 10, 2)->default(0);
            $table->decimal('total', 10, 2);

            $table->timestamps();

            $table->index('order_id');
            $table->index('inventory_item_id');
        });


        Schema::create('order_payments', function (Blueprint $table) {
            $table->id();

            $table->foreignId('order_id')
                ->constrained('orders')
                ->onDelete('cascade');

            $table->decimal('amount', 10, 2);
            $table->string('payment_method')->nullable(); // cash, card, bank, etc.
            $table->dateTime('paid_at')->default(DB::raw('CURRENT_TIMESTAMP'));
            $table->string('reference')->nullable();
            $table->text('notes')->nullable();

            $table->timestamps();

            $table->index('order_id');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('orders');
    }
};
