<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('installments', function (Blueprint $table) {
            $table->id();
            $table->foreignId('order_id')
                ->constrained('orders')
                ->onDelete('cascade');

            // Customer Information
            $table->string('customer_cnic', 20);
            $table->string('customer_address')->nullable();
            $table->string('customer_contact')->nullable();

            // Guarantor Information
            $table->string('guarantor_name');
            $table->string('guarantor_cnic', 20);
            $table->string('guarantor_contact');
            $table->string('guarantor_address')->nullable();
            $table->string('guarantor_relation')->nullable();

            // Financial Details
            $table->decimal('total_amount', 10, 2);
            $table->decimal('down_payment', 10, 2)->default(0);
            $table->decimal('financed_amount', 10, 2); // total - down_payment
            $table->decimal('markup_percentage', 5, 2)->default(0);
            $table->decimal('markup_amount', 10, 2)->default(0);
            $table->decimal('total_payable', 10, 2); // financed + markup
            
            // Installment Terms
            $table->integer('number_of_installments');
            $table->decimal('installment_amount', 10, 2);
            $table->enum('installment_frequency', ['daily', 'weekly', 'bi-weekly', 'monthly'])
                ->default('monthly');
            
            // Dates
            $table->date('start_date');
            $table->date('end_date')->nullable();
            
            // Status
            $table->enum('status', ['active', 'completed', 'defaulted', 'suspended', 'cancelled'])
                ->default('active');
            
            // Contract
            $table->timestamp('contract_generated_at')->nullable();
            $table->text('contract_path')->nullable();
            
            // Additional Info
            $table->text('terms_and_conditions')->nullable();
            $table->text('notes')->nullable();

            $table->timestamps();

            $table->index('order_id');
            $table->index('status');
            $table->index('customer_cnic');
            $table->index('start_date');
        });


         Schema::create('installment_documents', function (Blueprint $table) {
            $table->id();
            $table->foreignId('installment_id')
                ->constrained('installments')
                ->onDelete('cascade');

            $table->enum('document_type', [
                'customer_cnic',
                'guarantor_cnic',
                'utility_bill',
                'salary_slip',
                'bank_statement',
                'employment_letter',
                'other'
            ]);
            
            $table->string('file_name');
            $table->string('file_path');
            $table->string('mime_type')->nullable();
            $table->integer('file_size')->nullable(); // in bytes
            $table->text('description')->nullable();

            $table->timestamps();

            $table->index('installment_id');
            $table->index('document_type');
        });





        Schema::create('installment_payments', function (Blueprint $table) {
            $table->id();
            $table->foreignId('installment_id')
                ->constrained('installments')
                ->onDelete('cascade');

            $table->integer('installment_number'); // 1, 2, 3...
            $table->date('due_date');
            $table->decimal('amount_due', 10, 2);
            $table->decimal('amount_paid', 10, 2)->default(0);
            $table->date('paid_date')->nullable();
            
            $table->enum('status', ['pending', 'paid', 'overdue', 'partial'])
                ->default('pending');
            
            $table->decimal('late_fee', 10, 2)->default(0);
            $table->string('payment_method')->nullable(); // cash, bank, card
            $table->string('reference')->nullable();
            $table->text('notes')->nullable();

            $table->timestamps();

            $table->index('installment_id');
            $table->index('status');
            $table->index('due_date');
            $table->unique(['installment_id', 'installment_number']);
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('installments');
    }
};