<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('contract_templates', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->text('template_content'); // HTML content with placeholders
            $table->boolean('is_active')->default(true);
            $table->boolean('is_default')->default(false);
            $table->text('description')->nullable();
            $table->json('available_placeholders')->nullable(); // List of available placeholders
            $table->timestamps();

            $table->index('is_active');
            $table->index('is_default');
        });

        // Insert default template
        DB::table('contract_templates')->insert([
            'name' => 'Default Installment Contract',
            'is_active' => true,
            'is_default' => true,
            'template_content' => $this->getDefaultTemplate(),
            'available_placeholders' => json_encode([
                '{{contract_date}}',
                '{{customer_name}}',
                '{{customer_cnic}}',
                '{{customer_contact}}',
                '{{customer_address}}',
                '{{guarantor_name}}',
                '{{guarantor_cnic}}',
                '{{guarantor_contact}}',
                '{{guarantor_address}}',
                '{{guarantor_relation}}',
                '{{device_details}}',
                '{{total_amount}}',
                '{{down_payment}}',
                '{{financed_amount}}',
                '{{markup_percentage}}',
                '{{markup_amount}}',
                '{{total_payable}}',
                '{{number_of_installments}}',
                '{{installment_amount}}',
                '{{installment_frequency}}',
                '{{start_date}}',
                '{{order_number}}',
            ]),
            'created_at' => now(),
            'updated_at' => now(),
        ]);
    }

    private function getDefaultTemplate(): string
    {
        return <<<HTML
<!DOCTYPE html>
<html>
<head>
    <style>
        body { font-family: Arial, sans-serif; margin: 40px; }
        .header { text-align: center; margin-bottom: 30px; }
        .section { margin-bottom: 20px; }
        .label { font-weight: bold; }
        table { width: 100%; border-collapse: collapse; margin: 20px 0; }
        th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
        th { background-color: #f2f2f2; }
        .signature-section { margin-top: 50px; }
        .signature-box { display: inline-block; width: 45%; }
    </style>
</head>
<body>
    <div class="header">
        <h2>INSTALLMENT SALE AGREEMENT</h2>
        <p>Contract Date: {{contract_date}}</p>
        <p>Order Number: {{order_number}}</p>
    </div>

    <div class="section">
        <h3>BUYER INFORMATION</h3>
        <p><span class="label">Name:</span> {{customer_name}}</p>
        <p><span class="label">CNIC:</span> {{customer_cnic}}</p>
        <p><span class="label">Contact:</span> {{customer_contact}}</p>
        <p><span class="label">Address:</span> {{customer_address}}</p>
    </div>

    <div class="section">
        <h3>GUARANTOR INFORMATION</h3>
        <p><span class="label">Name:</span> {{guarantor_name}}</p>
        <p><span class="label">CNIC:</span> {{guarantor_cnic}}</p>
        <p><span class="label">Contact:</span> {{guarantor_contact}}</p>
        <p><span class="label">Address:</span> {{guarantor_address}}</p>
        <p><span class="label">Relation:</span> {{guarantor_relation}}</p>
    </div>

    <div class="section">
        <h3>PRODUCT DETAILS</h3>
        {{device_details}}
    </div>

    <div class="section">
        <h3>PAYMENT TERMS</h3>
        <table>
            <tr>
                <td class="label">Total Amount:</td>
                <td>Rs. {{total_amount}}</td>
            </tr>
            <tr>
                <td class="label">Down Payment:</td>
                <td>Rs. {{down_payment}}</td>
            </tr>
            <tr>
                <td class="label">Financed Amount:</td>
                <td>Rs. {{financed_amount}}</td>
            </tr>
            <tr>
                <td class="label">Markup ({{markup_percentage}}%):</td>
                <td>Rs. {{markup_amount}}</td>
            </tr>
            <tr>
                <td class="label">Total Payable:</td>
                <td>Rs. {{total_payable}}</td>
            </tr>
            <tr>
                <td class="label">Number of Installments:</td>
                <td>{{number_of_installments}} ({{installment_frequency}})</td>
            </tr>
            <tr>
                <td class="label">Installment Amount:</td>
                <td>Rs. {{installment_amount}}</td>
            </tr>
            <tr>
                <td class="label">Start Date:</td>
                <td>{{start_date}}</td>
            </tr>
        </table>
    </div>

    <div class="section">
        <h3>TERMS AND CONDITIONS</h3>
        <ol>
            <li>The buyer agrees to pay the installment amount on or before the due date.</li>
            <li>Late payment may incur additional charges.</li>
            <li>The seller retains ownership until full payment is received.</li>
            <li>The guarantor is liable in case of default by the buyer.</li>
            <li>This agreement is governed by the laws of Pakistan.</li>
        </ol>
    </div>

    <div class="signature-section">
        <div class="signature-box">
            <p>_________________________</p>
            <p><span class="label">Buyer Signature</span></p>
            <p>{{customer_name}}</p>
            <p>Date: _________________</p>
        </div>
        <div class="signature-box" style="float: right;">
            <p>_________________________</p>
            <p><span class="label">Guarantor Signature</span></p>
            <p>{{guarantor_name}}</p>
            <p>Date: _________________</p>
        </div>
    </div>

    <div style="clear: both; margin-top: 50px; text-align: center;">
        <p>_________________________</p>
        <p><span class="label">Seller Signature</span></p>
        <p>Date: _________________</p>
    </div>
</body>
</html>
HTML;
    }

    public function down(): void
    {
        Schema::dropIfExists('contract_templates');
    }
};