<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Create accounts table FIRST (since transactions references it)
        Schema::create('accounts', function (Blueprint $table) {
            $table->id();

            $table->string('name'); // "Main Cash", "HBL Bank Account", "JazzCash Wallet"
            $table->string('account_number')->nullable(); // Bank account number

            $table->enum('type', [
                'cash',           // Physical cash
                'bank',           // Bank account
                'mobile_money',   // JazzCash, EasyPaisa, etc.
                'credit_card',    // Credit card account
                'other'
            ]);

            $table->string('bank_name')->nullable(); // For bank accounts
            $table->string('branch')->nullable();
            $table->string('currency', 3)->default('PKR');

            $table->decimal('opening_balance', 15, 2)->default(0);
            $table->decimal('current_balance', 15, 2)->default(0);

            $table->boolean('is_active')->default(true);
            $table->boolean('is_default')->default(false); // Default account for transactions

            $table->text('description')->nullable();
            $table->timestamps();
            $table->softDeletes();
        });

        // Create transaction_categories table SECOND (since transactions references it)
        Schema::create('transaction_categories', function (Blueprint $table) {
            $table->id();

            $table->string('name'); // "Rent", "Utilities", "Salaries", "Office Supplies"
            $table->string('slug')->unique();

            $table->enum('type', ['income', 'expense']); // For reporting

            $table->foreignId('parent_id')->nullable()->constrained('transaction_categories')->nullOnDelete();

            $table->text('description')->nullable();
            $table->string('color', 7)->nullable(); // Hex color for UI
            $table->string('icon')->nullable(); // Icon name

            $table->boolean('is_active')->default(true);
            $table->timestamps();
        });

        // Create transactions table LAST (after its dependencies exist)
        Schema::create('transactions', function (Blueprint $table) {
            $table->id();

            // Transaction identification
            $table->string('transaction_number')->unique(); // TXN-0001
            $table->enum('type', [
                'sale_payment',       // Customer pays for sale/order
                'purchase_payment',   // You pay supplier for purchase
                'installment_payment', // Customer pays installment
                'expense',            // Business expense
                'income',             // Other income
                'refund',             // Refund to customer
                'transfer',           // Transfer between accounts
                'adjustment',         // Manual adjustment
                'opening_balance',    // Opening balance for account
            ]);

            // Amount and direction
            $table->enum('direction', ['credit', 'debit']); // credit = money IN, debit = money OUT
            $table->decimal('amount', 15, 2);

            // Related entity (polymorphic - can link to any module)
            $table->string('related_type')->nullable(); // 'App\Models\Order', 'App\Models\Purchase', etc.
            $table->unsignedBigInteger('related_id')->nullable();
            $table->index(['related_type', 'related_id']); // For quick lookups

            // Payment details
            $table->enum('payment_method', [
                'cash',
                'bank_transfer',
                'credit_card',
                'debit_card',
                'mobile_money',
                'cheque',
                'online',
                'other'
            ]);

            // Account/Category tracking
            $table->foreignId('account_id')->nullable()->constrained('accounts')->nullOnDelete();
            $table->foreignId('category_id')->nullable()->constrained('transaction_categories')->nullOnDelete();

            // Party information (who you're transacting with)
            $table->string('party_type')->nullable(); // 'customer', 'supplier', 'employee', 'other'
            $table->unsignedBigInteger('party_id')->nullable(); // ID of customer/supplier/employee

            // Date and reference
            $table->date('transaction_date');
            $table->string('reference_number')->nullable(); // Bank ref, cheque no, etc.

            // Status
            $table->enum('status', ['completed', 'pending', 'failed', 'cancelled'])->default('completed');

            // Additional details
            $table->text('description')->nullable();
            $table->text('notes')->nullable();

            // Attachments (receipt, invoice, etc.)
            $table->json('attachments')->nullable(); // Array of file paths

            // For reconciliation
            $table->boolean('is_reconciled')->default(false);
            $table->date('reconciled_date')->nullable();

            // Audit
            $table->foreignId('created_by')->nullable()->constrained('users')->nullOnDelete();
            $table->timestamps();
            $table->softDeletes();

            // Indexes for performance
            $table->index('transaction_date');
            $table->index('type');
            $table->index('direction');
            $table->index('payment_method');
            $table->index('status');
            $table->index(['party_type', 'party_id']);
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('transactions');
        Schema::dropIfExists('transaction_categories');
        Schema::dropIfExists('accounts');
    }
};
